# UI Component Generator - Initialization Manifest

## Skill Identity
- **Name**: UI Component Generator
- **Category**: design-media
- **Version**: 1.0.0
- **Created**: 2025-08-22
- **Difficulty**: intermediate
- **Platforms**: claude, chatgpt, gemini, copilot

## Skill Description
An AI skill that generates production-ready UI components from natural language descriptions with clean, accessible code. Supports three output formats: Vanilla HTML/CSS with CSS custom properties, Tailwind CSS, and React TypeScript. Every component includes built-in accessibility (ARIA, keyboard navigation), responsive design, dark mode support, and follows WCAG 2.1 AA standards.

## Target Users
- Frontend developers building design systems
- UI designers creating interactive prototypes
- Teams needing consistent component patterns
- Developers learning accessibility best practices
- Rapid prototyping and MVP development teams

## Installation & Setup

### Method 1: Direct Copy (Recommended)
1. Copy the complete skill prompt from the "Complete Skill Prompt" section below
2. Paste into your AI assistant (Claude, ChatGPT, Gemini, or Copilot)
3. Start requesting components immediately

### Method 2: Save as Custom Instruction
1. In your AI assistant settings, navigate to custom instructions or system prompts
2. Create a new custom instruction named "UI Component Generator"
3. Paste the complete skill prompt
4. Save and activate

### Method 3: Project-Specific Integration
1. Create a `.ai-skills/` directory in your project root
2. Save the skill prompt as `ui-component-generator.md`
3. Reference it when needed: "Using the UI Component Generator skill, create..."

## Complete Skill Prompt

```
You are an expert UI component generator. Generate production-ready UI components from natural language descriptions with clean, accessible code.

## Supported Component Types
- Buttons (primary, secondary, ghost, icon, loading)
- Cards (content, pricing, profile, feature, testimonial)
- Forms (input, textarea, select, checkbox, radio, switch)
- Modals/dialogs (alert, confirm, form, fullscreen)
- Navigation (navbar, sidebar, breadcrumbs, tabs, pagination)
- Alerts/notifications (info, success, warning, error, toast)
- Empty states (no data, error, search, onboarding)
- Loading states (spinner, skeleton, progress bar, shimmer)

## Output Formats
When user requests a component, ask which format they prefer:
1. **Vanilla HTML/CSS** - Pure HTML with CSS custom properties
2. **Tailwind CSS** - HTML with Tailwind utility classes
3. **React TypeScript** - React component with TypeScript types

## Component Requirements
Every component you generate MUST include:

### Accessibility
- Semantic HTML elements (button, nav, main, aside, etc.)
- ARIA attributes (aria-label, aria-describedby, aria-expanded, role)
- Keyboard navigation support (Tab, Enter, Escape, Arrow keys)
- Focus indicators (visible focus rings, focus-within states)
- Screen reader text where needed (sr-only class)

### Visual States
- Default state styling
- Hover states (cursor: pointer, background changes)
- Active/pressed states
- Focus states (outline or ring)
- Disabled states (reduced opacity, cursor: not-allowed)
- Loading states where applicable

### Responsive Design
- Mobile-first approach (min-width media queries)
- Touch-friendly sizes (44px minimum touch targets)
- Responsive typography (clamp() or fluid scale)
- Breakpoints: sm (640px), md (768px), lg (1024px), xl (1280px)

### Dark Mode
- CSS custom properties for colors OR
- Tailwind dark: variants OR
- React context/theme switching
- Proper contrast ratios (WCAG AA minimum: 4.5:1 for text, 3:1 for large text)

## Generation Workflow

### Step 1: Clarify Requirements
When user requests a component, ask:
- "Which format: Vanilla HTML/CSS, Tailwind, or React TypeScript?"
- "Do you need dark mode support?" (default: yes)
- "Any specific colors or brand guidelines?" (default: use neutral palette)
- "Special behavior or variants needed?"

### Step 2: Generate Component Code
Provide complete, copy-paste ready code.

### Step 3: Provide Usage Examples
After generating component, provide 2-3 usage examples showing:
- Basic usage
- Different variants/states
- Integration patterns

### Step 4: List Customization Options
Explain what can be customized:
- Color scheme (CSS variables or Tailwind config)
- Sizing scale
- Border radius
- Animation duration
- Breakpoints

## Anti-Patterns to Avoid
- ❌ Using div instead of semantic elements (button, nav, article)
- ❌ Missing ARIA labels on icon-only buttons
- ❌ Click handlers on non-interactive elements without role="button"
- ❌ Hardcoded colors without dark mode consideration
- ❌ Inaccessible color contrast ratios
- ❌ Missing focus indicators
- ❌ Non-responsive touch targets (<44px)
- ❌ Inline styles instead of classes
- ❌ Using !important (use specificity properly)

## Response Format
Structure your response as:

1. **Component Preview** (describe what it looks like)
2. **Code** (complete implementation)
3. **Usage Examples** (2-3 examples)
4. **Customization Guide** (what can be tweaked)
5. **Accessibility Notes** (keyboard shortcuts, screen reader behavior)

Now generate production-ready UI components based on user descriptions!
```

## Example Usage Scenarios

### Scenario 1: Creating a Button Component
**User Input:**
```
Create a primary button component in React TypeScript with loading state
```

**Expected AI Response:**
- Asks clarifying questions about dark mode and colors
- Generates complete React TypeScript component with ButtonProps interface
- Includes loading spinner SVG animation
- Shows usage examples (basic, with icon, loading state)
- Lists customization options
- Documents keyboard navigation and screen reader behavior

### Scenario 2: Building a Pricing Card
**User Input:**
```
I need a pricing card in Tailwind CSS with a popular badge
```

**Expected AI Response:**
- Generates HTML with Tailwind classes
- Includes popular badge with contrasting colors
- Shows responsive grid layout
- Provides dark mode variants (dark: classes)
- Demonstrates 3 usage examples (basic, featured, enterprise tiers)
- Explains how to customize colors via tailwind.config.js

### Scenario 3: Form Input with Validation
**User Input:**
```
Generate a text input with validation states in vanilla HTML/CSS
```

**Expected AI Response:**
- Creates semantic HTML with label, input, error message
- Uses CSS custom properties for theming
- Includes error, success, and default states
- Shows ARIA attributes for error messaging
- Provides @media (prefers-color-scheme: dark) styles
- Documents focus management for screen readers

## Code Examples

### Example 1: Button Component (React TypeScript)

```typescript
import React from 'react';

interface ButtonProps extends React.ButtonHTMLAttributes<HTMLButtonElement> {
  variant?: 'primary' | 'secondary' | 'ghost';
  size?: 'sm' | 'md' | 'lg';
  isLoading?: boolean;
  leftIcon?: React.ReactNode;
  rightIcon?: React.ReactNode;
}

export const Button: React.FC<ButtonProps> = ({
  variant = 'primary',
  size = 'md',
  isLoading = false,
  leftIcon,
  rightIcon,
  children,
  disabled,
  className = '',
  ...props
}) => {
  const baseStyles = `
    inline-flex items-center justify-center gap-2
    font-medium rounded-lg
    transition-all duration-200
    focus:outline-none focus:ring-2 focus:ring-offset-2
  `;

  const variantStyles = {
    primary: `
      bg-blue-600 hover:bg-blue-700 active:bg-blue-800
      text-white
      focus:ring-blue-500
      dark:bg-blue-500 dark:hover:bg-blue-600
    `,
    secondary: `
      bg-gray-200 hover:bg-gray-300 active:bg-gray-400
      text-gray-900
      focus:ring-gray-500
      dark:bg-gray-700 dark:hover:bg-gray-600 dark:text-white
    `,
    ghost: `
      bg-transparent hover:bg-gray-100 active:bg-gray-200
      text-gray-700
      focus:ring-gray-500
      dark:hover:bg-gray-800 dark:text-gray-300
    `
  };

  const sizeStyles = {
    sm: 'px-3 py-1.5 text-sm min-h-[36px]',
    md: 'px-4 py-2 text-base min-h-[44px]',
    lg: 'px-6 py-3 text-lg min-h-[52px]'
  };

  const combinedClasses = `
    ${baseStyles}
    ${variantStyles[variant]}
    ${sizeStyles[size]}
    ${(disabled || isLoading) ? 'opacity-50 cursor-not-allowed' : ''}
    ${className}
  `.trim().replace(/\s+/g, ' ');

  return (
    <button
      className={combinedClasses}
      disabled={disabled || isLoading}
      {...props}
    >
      {isLoading ? (
        <>
          <svg
            className="animate-spin h-4 w-4"
            xmlns="http://www.w3.org/2000/svg"
            fill="none"
            viewBox="0 0 24 24"
            aria-hidden="true"
          >
            <circle
              className="opacity-25"
              cx="12"
              cy="12"
              r="10"
              stroke="currentColor"
              strokeWidth="4"
            />
            <path
              className="opacity-75"
              fill="currentColor"
              d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
            />
          </svg>
          <span>Loading...</span>
        </>
      ) : (
        <>
          {leftIcon && <span aria-hidden="true">{leftIcon}</span>}
          {children}
          {rightIcon && <span aria-hidden="true">{rightIcon}</span>}
        </>
      )}
    </button>
  );
};

Button.displayName = 'Button';
```

**Usage:**
```typescript
// Basic
<Button>Click Me</Button>

// With variant and size
<Button variant="secondary" size="lg">Large Secondary</Button>

// Loading state
<Button isLoading disabled>Submitting...</Button>

// With icons
<Button leftIcon={<CheckIcon />} variant="primary">
  Confirm
</Button>
```

### Example 2: Card Component (Tailwind CSS)

```html
<!-- Pricing Card with Popular Badge -->
<article
  class="relative flex flex-col p-6 bg-white dark:bg-gray-800 rounded-xl shadow-lg hover:shadow-xl transition-shadow duration-300 border border-gray-200 dark:border-gray-700"
  role="article"
  aria-labelledby="pricing-title"
>
  <!-- Popular Badge -->
  <div
    class="absolute -top-4 left-1/2 -translate-x-1/2 px-4 py-1 bg-blue-600 dark:bg-blue-500 text-white text-sm font-semibold rounded-full shadow-md"
    aria-label="Most popular plan"
  >
    Most Popular
  </div>

  <!-- Card Header -->
  <div class="text-center mb-6 mt-2">
    <h3
      id="pricing-title"
      class="text-2xl font-bold text-gray-900 dark:text-white mb-2"
    >
      Pro Plan
    </h3>
    <div class="flex items-baseline justify-center gap-1">
      <span class="text-4xl font-extrabold text-gray-900 dark:text-white">$49</span>
      <span class="text-gray-500 dark:text-gray-400">/month</span>
    </div>
  </div>

  <!-- Features List -->
  <ul class="flex-1 space-y-3 mb-6" role="list">
    <li class="flex items-start gap-3">
      <svg class="w-5 h-5 text-green-500 flex-shrink-0 mt-0.5" aria-hidden="true" fill="currentColor" viewBox="0 0 20 20">
        <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"/>
      </svg>
      <span class="text-gray-700 dark:text-gray-300">Unlimited projects</span>
    </li>
    <li class="flex items-start gap-3">
      <svg class="w-5 h-5 text-green-500 flex-shrink-0 mt-0.5" aria-hidden="true" fill="currentColor" viewBox="0 0 20 20">
        <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"/>
      </svg>
      <span class="text-gray-700 dark:text-gray-300">Priority support</span>
    </li>
    <li class="flex items-start gap-3">
      <svg class="w-5 h-5 text-green-500 flex-shrink-0 mt-0.5" aria-hidden="true" fill="currentColor" viewBox="0 0 20 20">
        <path fill-rule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clip-rule="evenodd"/>
      </svg>
      <span class="text-gray-700 dark:text-gray-300">Advanced analytics</span>
    </li>
  </ul>

  <!-- CTA Button -->
  <button
    class="w-full px-6 py-3 bg-blue-600 hover:bg-blue-700 active:bg-blue-800 text-white font-semibold rounded-lg transition-colors duration-200 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 dark:bg-blue-500 dark:hover:bg-blue-600 min-h-[48px]"
    aria-label="Get started with Pro plan"
  >
    Get Started
  </button>
</article>
```

### Example 3: Form Input (Vanilla HTML/CSS)

```html
<div class="form-field">
  <label for="email" class="form-label">
    Email Address
    <span class="required" aria-label="required">*</span>
  </label>

  <input
    id="email"
    type="email"
    class="form-input"
    placeholder="you@example.com"
    aria-describedby="email-error email-hint"
    aria-invalid="true"
    aria-required="true"
  >

  <span id="email-hint" class="form-hint">
    We'll never share your email with anyone else.
  </span>

  <span id="email-error" class="form-error" role="alert">
    <svg class="error-icon" aria-hidden="true" viewBox="0 0 20 20" fill="currentColor">
      <path fill-rule="evenodd" d="M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7 4a1 1 0 11-2 0 1 1 0 012 0zm-1-9a1 1 0 00-1 1v4a1 1 0 102 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/>
    </svg>
    Please enter a valid email address
  </span>
</div>

<style>
:root {
  /* Colors - Light Mode */
  --color-bg-base: #ffffff;
  --color-text-primary: #1f2937;
  --color-text-secondary: #6b7280;
  --color-border: #d1d5db;
  --color-border-focus: #3b82f6;
  --color-error: #ef4444;
  --color-error-bg: #fef2f2;

  /* Spacing */
  --space-1: 0.25rem;
  --space-2: 0.5rem;
  --space-3: 0.75rem;
  --space-4: 1rem;

  /* Typography */
  --text-sm: 0.875rem;
  --text-base: 1rem;

  /* Radius */
  --radius-md: 0.5rem;

  /* Shadows */
  --shadow-sm: 0 1px 2px 0 rgb(0 0 0 / 0.05);
  --shadow-focus: 0 0 0 3px rgb(59 130 246 / 0.3);
}

@media (prefers-color-scheme: dark) {
  :root {
    --color-bg-base: #1f2937;
    --color-text-primary: #f9fafb;
    --color-text-secondary: #9ca3af;
    --color-border: #374151;
    --color-border-focus: #60a5fa;
    --color-error: #f87171;
    --color-error-bg: #7f1d1d;
  }
}

.form-field {
  display: flex;
  flex-direction: column;
  gap: var(--space-2);
  width: 100%;
}

.form-label {
  font-size: var(--text-base);
  font-weight: 500;
  color: var(--color-text-primary);
}

.required {
  color: var(--color-error);
  margin-left: var(--space-1);
}

.form-input {
  width: 100%;
  min-height: 44px;
  padding: var(--space-2) var(--space-3);
  font-size: var(--text-base);
  color: var(--color-text-primary);
  background-color: var(--color-bg-base);
  border: 2px solid var(--color-border);
  border-radius: var(--radius-md);
  transition: all 0.2s ease;
}

.form-input::placeholder {
  color: var(--color-text-secondary);
}

.form-input:hover:not(:disabled) {
  border-color: var(--color-border-focus);
}

.form-input:focus {
  outline: none;
  border-color: var(--color-border-focus);
  box-shadow: var(--shadow-focus);
}

.form-input[aria-invalid="true"] {
  border-color: var(--color-error);
}

.form-input[aria-invalid="true"]:focus {
  box-shadow: 0 0 0 3px rgb(239 68 68 / 0.3);
}

.form-input:disabled {
  opacity: 0.5;
  cursor: not-allowed;
  background-color: var(--color-text-secondary);
}

.form-hint {
  font-size: var(--text-sm);
  color: var(--color-text-secondary);
}

.form-error {
  display: flex;
  align-items-center;
  gap: var(--space-2);
  font-size: var(--text-sm);
  color: var(--color-error);
  padding: var(--space-2) var(--space-3);
  background-color: var(--color-error-bg);
  border-radius: var(--radius-md);
}

.error-icon {
  width: 1rem;
  height: 1rem;
  flex-shrink: 0;
}

/* Hide error message when input is valid */
.form-input:not([aria-invalid="true"]) ~ .form-error {
  display: none;
}

/* Responsive */
@media (min-width: 640px) {
  .form-field {
    max-width: 400px;
  }
}
</style>
```

## Customization Guide

### Changing Colors
**Vanilla CSS:**
```css
:root {
  --color-primary: #your-brand-color;
  --color-primary-hover: #darker-shade;
}
```

**Tailwind CSS:**
```javascript
// tailwind.config.js
module.exports = {
  theme: {
    extend: {
      colors: {
        primary: {
          50: '#...',
          // ... full scale
          950: '#...'
        }
      }
    }
  }
}
```

**React TypeScript:**
```typescript
// Create a theme context
const theme = {
  colors: {
    primary: '#your-brand-color'
  }
};
```

### Adjusting Sizes
Modify spacing tokens or size props to change component dimensions globally or per-instance.

### Animation Speed
Change transition duration from 200ms to your preference (150ms for snappy, 300ms for smooth).

## Best Practices

1. **Always Use Semantic HTML**: `<button>` for buttons, `<nav>` for navigation, `<article>` for cards
2. **Include ARIA Attributes**: Only when they add value, don't over-use
3. **Test Keyboard Navigation**: Tab through your UI, ensure all interactive elements are reachable
4. **Check Color Contrast**: Use tools like WebAIM Contrast Checker
5. **Mobile-First Responsive**: Start with mobile layout, enhance for larger screens
6. **Dark Mode by Default**: Include dark mode in initial implementation
7. **Touch-Friendly Targets**: 44px × 44px minimum for buttons and interactive elements
8. **Loading States**: Show feedback for async operations
9. **Error States**: Clearly indicate validation errors with color AND text

## Troubleshooting

### Component Not Accessible
- Check if semantic HTML is used (`<button>` vs `<div onClick>`)
- Verify ARIA attributes are present (aria-label on icon buttons)
- Test with keyboard (Tab, Enter, Escape should work)
- Run axe DevTools or WAVE browser extension

### Dark Mode Not Working
- **Vanilla CSS**: Check `@media (prefers-color-scheme: dark)` is present
- **Tailwind**: Verify `dark:` classes and darkMode config in tailwind.config.js
- **React**: Ensure theme context is wrapped around component tree

### Styles Not Applying
- Check for CSS specificity conflicts
- Verify class names match between HTML and CSS
- Ensure CSS custom properties are defined in `:root`
- For Tailwind, make sure JIT mode is enabled and build process is running

### TypeScript Errors
- Verify React.ButtonHTMLAttributes is extended for button props
- Check that all required props have default values
- Ensure event handler types match (React.MouseEvent<HTMLButtonElement>)

## Additional Resources

### Accessibility References
- [WCAG 2.1 Guidelines](https://www.w3.org/WAI/WCAG21/quickref/)
- [ARIA Authoring Practices Guide](https://www.w3.org/WAI/ARIA/apg/)
- [WebAIM Contrast Checker](https://webaim.org/resources/contrastchecker/)

### Design Systems
- [Tailwind UI Components](https://tailwindui.com/components)
- [Radix UI Primitives](https://www.radix-ui.com/)
- [Headless UI](https://headlessui.com/)
- [shadcn/ui](https://ui.shadcn.com/)

### Testing Tools
- [axe DevTools](https://www.deque.com/axe/devtools/)
- [WAVE Browser Extension](https://wave.webaim.org/extension/)
- [React Testing Library](https://testing-library.com/docs/react-testing-library/intro/)

## Version History

- **v1.0.0** (2025-08-22): Initial release
  - 8 component categories supported
  - 3 output formats (Vanilla, Tailwind, React TS)
  - WCAG 2.1 AA accessibility compliance
  - Dark mode support
  - Responsive design patterns

## License & Attribution
This skill is provided as-is for use with any AI assistant. No attribution required. Feel free to modify and adapt for your specific needs.

## Support & Community
For issues, suggestions, or contributions, visit the AI Skills Directory repository or community forums.

---

**Quick Start Reminder:**
1. Copy the "Complete Skill Prompt" section above
2. Paste into your AI assistant
3. Request: "Create a [component type] in [format]"
4. Enjoy production-ready, accessible components!
