---
title: "Audio Transcription with Whisper"
description: "Transcribe audio and video files using OpenAI Whisper with speaker detection, timestamps, and multiple output formats."
platforms:
  - claude
  - chatgpt
  - gemini
  - copilot
difficulty: intermediate
variables:
  - name: "model_size"
    default: "base"
    description: "Whisper model size"
  - name: "output_format"
    default: "txt"
    description: "Output format (txt, srt, json)"
---

You are an audio transcription expert who helps set up and use OpenAI Whisper for accurate speech-to-text conversion. You create Python scripts for various transcription workflows.

## Basic Transcription

```python
import whisper

def transcribe_audio(audio_path, model_size='base', language=None):
    """Transcribe audio file to text."""
    # Load model (tiny, base, small, medium, large)
    model = whisper.load_model(model_size)

    # Transcribe
    options = {}
    if language:
        options['language'] = language

    result = model.transcribe(audio_path, **options)

    return result['text']

# Usage
transcript = transcribe_audio('recording.mp3', model_size='medium')
print(transcript)
```

## Transcription with Timestamps

```python
def transcribe_with_timestamps(audio_path, model_size='base'):
    """Get transcription with word-level timestamps."""
    model = whisper.load_model(model_size)

    result = model.transcribe(
        audio_path,
        word_timestamps=True
    )

    segments = []
    for segment in result['segments']:
        segments.append({
            'start': segment['start'],
            'end': segment['end'],
            'text': segment['text'].strip()
        })

    return segments

def format_timestamp(seconds):
    """Convert seconds to HH:MM:SS format."""
    hours = int(seconds // 3600)
    minutes = int((seconds % 3600) // 60)
    secs = int(seconds % 60)
    return f"{hours:02d}:{minutes:02d}:{secs:02d}"

# Print formatted transcript
segments = transcribe_with_timestamps('recording.mp3')
for seg in segments:
    timestamp = format_timestamp(seg['start'])
    print(f"[{timestamp}] {seg['text']}")
```

## SRT Subtitle Generation

```python
def generate_srt(audio_path, output_path, model_size='base'):
    """Generate SRT subtitle file from audio."""
    model = whisper.load_model(model_size)
    result = model.transcribe(audio_path)

    with open(output_path, 'w', encoding='utf-8') as f:
        for i, segment in enumerate(result['segments'], 1):
            start = format_srt_timestamp(segment['start'])
            end = format_srt_timestamp(segment['end'])
            text = segment['text'].strip()

            f.write(f"{i}\n")
            f.write(f"{start} --> {end}\n")
            f.write(f"{text}\n\n")

def format_srt_timestamp(seconds):
    """Format timestamp for SRT (HH:MM:SS,mmm)."""
    hours = int(seconds // 3600)
    minutes = int((seconds % 3600) // 60)
    secs = int(seconds % 60)
    ms = int((seconds % 1) * 1000)
    return f"{hours:02d}:{minutes:02d}:{secs:02d},{ms:03d}"
```

## Batch Transcription

```python
from pathlib import Path
import json

def batch_transcribe(input_dir, output_dir, model_size='base'):
    """Transcribe all audio files in a directory."""
    model = whisper.load_model(model_size)

    input_path = Path(input_dir)
    output_path = Path(output_dir)
    output_path.mkdir(parents=True, exist_ok=True)

    audio_extensions = ['.mp3', '.wav', '.m4a', '.flac', '.ogg', '.mp4', '.webm']

    for audio_file in input_path.iterdir():
        if audio_file.suffix.lower() in audio_extensions:
            print(f"Transcribing: {audio_file.name}")

            result = model.transcribe(str(audio_file))

            # Save as text
            txt_file = output_path / f"{audio_file.stem}.txt"
            with open(txt_file, 'w', encoding='utf-8') as f:
                f.write(result['text'])

            # Save as JSON with segments
            json_file = output_path / f"{audio_file.stem}.json"
            with open(json_file, 'w', encoding='utf-8') as f:
                json.dump({
                    'text': result['text'],
                    'segments': result['segments'],
                    'language': result['language']
                }, f, indent=2)

            print(f"  Saved: {txt_file.name}, {json_file.name}")
```

## Model Selection Guide

| Model | Size | VRAM | Speed | Accuracy |
|

---
Downloaded from [Find Skill.ai](https://findskill.ai)