# Accessible Color Palette Generator - Initialization Manifest

## Skill Metadata

**Name:** Accessible Color Palette Generator
**Category:** Design & Media
**Version:** 1.0.0
**Difficulty:** Intermediate
**Platforms:** Claude, ChatGPT, Gemini, Copilot
**Purpose:** Generate WCAG-compliant color palettes with guaranteed contrast ratios for accessible UI design

## Overview

This skill enables AI assistants to create production-ready, accessible color systems from a single brand color. It generates complete color scales for semantic roles (primary, success, warning, error, info, neutral), validates WCAG 2.1 compliance, provides dark mode variants, and exports in multiple formats (CSS, Tailwind, SCSS, JSON).

## Installation Instructions

### For Claude Code / Claude Desktop

1. Create skill directory:
   ```bash
   mkdir -p ~/.claude/skills/accessible-color-palette-generator
   ```

2. Save the skill instructions to:
   ```
   ~/.claude/skills/accessible-color-palette-generator/SKILL.md
   ```

3. Reference in conversation:
   ```
   @accessible-color-palette-generator Generate a palette from #3B82F6
   ```

### For ChatGPT Custom GPT

1. Create a new Custom GPT
2. Copy the entire skill content from the "Skill Instructions" section below
3. Paste into the Custom GPT instructions field
4. Add knowledge files (optional): color-theory.pdf, wcag-guidelines.pdf

### For Cursor / Continue

1. Add to `.cursorrules` or `.continueignore`:
   ```
   skills/accessible-color-palette-generator
   ```

2. Create skill file in project:
   ```bash
   mkdir -p .claude/skills
   cp SKILL.md .claude/skills/accessible-color-palette-generator.md
   ```

### Standalone Usage

Copy the skill prompt directly into any AI chat interface.

## Core Algorithms

### 1. Color Scale Generation Algorithm

```javascript
/**
 * Generate 11-step color scale (50-950) from base HSL color
 */
function generateColorScale(baseHSL, style = 'vibrant') {
  const { h, s, l } = baseHSL;

  // Style-based saturation adjustment
  const saturationMultipliers = {
    vibrant: 1.0,
    muted: 0.6,
    earthy: 0.7,
    pastel: 0.6,
    bold: 1.1,
    minimal: 0.3
  };

  const satAdjust = saturationMultipliers[style] || 1.0;
  const adjustedSat = Math.min(100, s * satAdjust);

  // Generate scale
  const scale = {
    50: { h, s: adjustedSat, l: 95 },
    100: { h, s: adjustedSat, l: 90 },
    200: { h, s: adjustedSat, l: 80 },
    300: { h, s: adjustedSat, l: 70 },
    400: { h, s: adjustedSat, l: 60 },
    500: { h, s: adjustedSat, l: 50 }, // Base
    600: { h, s: adjustedSat, l: 40 },
    700: { h, s: adjustedSat, l: 30 },
    800: { h, s: adjustedSat, l: 20 },
    900: { h, s: adjustedSat, l: 15 },
    950: { h, s: adjustedSat, l: 10 }
  };

  return scale;
}
```

### 2. WCAG Contrast Ratio Calculation

```javascript
/**
 * Calculate WCAG 2.1 contrast ratio between two colors
 */
function getContrastRatio(color1RGB, color2RGB) {
  const l1 = getRelativeLuminance(color1RGB);
  const l2 = getRelativeLuminance(color2RGB);

  const lighter = Math.max(l1, l2);
  const darker = Math.min(l1, l2);

  return (lighter + 0.05) / (darker + 0.05);
}

/**
 * Calculate relative luminance for RGB color
 */
function getRelativeLuminance(rgb) {
  // Normalize RGB values to 0-1
  const [r, g, b] = rgb.map(val => {
    const normalized = val / 255;
    return normalized <= 0.03928
      ? normalized / 12.92
      : Math.pow((normalized + 0.055) / 1.055, 2.4);
  });

  // Apply luminance formula
  return 0.2126 * r + 0.7152 * g + 0.0722 * b;
}

/**
 * Check WCAG compliance level
 */
function checkWCAGCompliance(ratio, level = 'AA') {
  const standards = {
    AA: {
      normalText: 4.5,
      largeText: 3.0,
      uiComponents: 3.0
    },
    AAA: {
      normalText: 7.0,
      largeText: 4.5,
      uiComponents: 3.0
    }
  };

  const required = standards[level];

  return {
    normalText: ratio >= required.normalText,
    largeText: ratio >= required.largeText,
    uiComponents: ratio >= required.uiComponents,
    ratio: ratio.toFixed(1)
  };
}
```

### 3. Color Space Conversion

```javascript
/**
 * Convert hex to HSL
 */
function hexToHSL(hex) {
  // Remove # if present
  hex = hex.replace(/^#/, '');

  // Parse RGB
  const r = parseInt(hex.substr(0, 2), 16) / 255;
  const g = parseInt(hex.substr(2, 2), 16) / 255;
  const b = parseInt(hex.substr(4, 2), 16) / 255;

  const max = Math.max(r, g, b);
  const min = Math.min(r, g, b);
  const diff = max - min;

  let h = 0;
  let s = 0;
  let l = (max + min) / 2;

  if (diff !== 0) {
    s = l > 0.5 ? diff / (2 - max - min) : diff / (max + min);

    switch (max) {
      case r:
        h = ((g - b) / diff + (g < b ? 6 : 0)) / 6;
        break;
      case g:
        h = ((b - r) / diff + 2) / 6;
        break;
      case b:
        h = ((r - g) / diff + 4) / 6;
        break;
    }
  }

  return {
    h: Math.round(h * 360),
    s: Math.round(s * 100),
    l: Math.round(l * 100)
  };
}

/**
 * Convert HSL to hex
 */
function hslToHex(h, s, l) {
  s /= 100;
  l /= 100;

  const c = (1 - Math.abs(2 * l - 1)) * s;
  const x = c * (1 - Math.abs((h / 60) % 2 - 1));
  const m = l - c / 2;

  let r = 0, g = 0, b = 0;

  if (h >= 0 && h < 60) {
    r = c; g = x; b = 0;
  } else if (h >= 60 && h < 120) {
    r = x; g = c; b = 0;
  } else if (h >= 120 && h < 180) {
    r = 0; g = c; b = x;
  } else if (h >= 180 && h < 240) {
    r = 0; g = x; b = c;
  } else if (h >= 240 && h < 300) {
    r = x; g = 0; b = c;
  } else if (h >= 300 && h < 360) {
    r = c; g = 0; b = x;
  }

  const toHex = (val) => {
    const hex = Math.round((val + m) * 255).toString(16);
    return hex.length === 1 ? '0' + hex : hex;
  };

  return `#${toHex(r)}${toHex(g)}${toHex(b)}`.toUpperCase();
}

/**
 * Convert HSL to RGB
 */
function hslToRGB(h, s, l) {
  s /= 100;
  l /= 100;

  const c = (1 - Math.abs(2 * l - 1)) * s;
  const x = c * (1 - Math.abs((h / 60) % 2 - 1));
  const m = l - c / 2;

  let r = 0, g = 0, b = 0;

  if (h >= 0 && h < 60) {
    r = c; g = x; b = 0;
  } else if (h >= 60 && h < 120) {
    r = x; g = c; b = 0;
  } else if (h >= 120 && h < 180) {
    r = 0; g = c; b = x;
  } else if (h >= 180 && h < 240) {
    r = 0; g = x; b = c;
  } else if (h >= 240 && h < 300) {
    r = x; g = 0; b = c;
  } else if (h >= 300 && h < 360) {
    r = c; g = 0; b = x;
  }

  return [
    Math.round((r + m) * 255),
    Math.round((g + m) * 255),
    Math.round((b + m) * 255)
  ];
}
```

### 4. Optimal Text Color Determination

```javascript
/**
 * Determine optimal text color (light or dark) for given background
 */
function getOptimalTextColor(bgHSL, wcagLevel = 'AA') {
  const bgRGB = hslToRGB(bgHSL.h, bgHSL.s, bgHSL.l);

  // Test both white and dark text
  const whiteRGB = [255, 255, 255];
  const darkRGB = [10, 10, 10]; // Near black

  const whiteContrast = getContrastRatio(bgRGB, whiteRGB);
  const darkContrast = getContrastRatio(bgRGB, darkRGB);

  const requiredRatio = wcagLevel === 'AAA' ? 7.0 : 4.5;

  // Prefer the option with higher contrast
  if (whiteContrast >= requiredRatio && darkContrast >= requiredRatio) {
    // Both pass, choose higher contrast
    return whiteContrast > darkContrast
      ? { color: 'hsl(0, 0%, 100%)', contrast: whiteContrast }
      : { color: 'hsl(0, 0%, 10%)', contrast: darkContrast };
  } else if (whiteContrast >= requiredRatio) {
    return { color: 'hsl(0, 0%, 100%)', contrast: whiteContrast };
  } else if (darkContrast >= requiredRatio) {
    return { color: 'hsl(0, 0%, 10%)', contrast: darkContrast };
  } else {
    // Neither passes, return higher contrast with warning
    return whiteContrast > darkContrast
      ? { color: 'hsl(0, 0%, 100%)', contrast: whiteContrast, warning: true }
      : { color: 'hsl(0, 0%, 10%)', contrast: darkContrast, warning: true };
  }
}
```

### 5. Semantic Color Generation

```javascript
/**
 * Generate semantic color palette from primary color
 */
function generateSemanticColors(primaryHSL, style) {
  return {
    primary: generateColorScale(primaryHSL, style),

    secondary: generateColorScale({
      h: (primaryHSL.h + 180) % 360, // Complementary
      s: primaryHSL.s * 0.9,
      l: primaryHSL.l
    }, style),

    success: generateColorScale({
      h: 142, // Green
      s: style === 'muted' ? 50 : 71,
      l: 45
    }, style),

    warning: generateColorScale({
      h: 38, // Amber
      s: style === 'muted' ? 70 : 92,
      l: 50
    }, style),

    error: generateColorScale({
      h: 0, // Red
      s: style === 'muted' ? 60 : 84,
      l: 50
    }, style),

    info: generateColorScale({
      h: 200, // Blue
      s: style === 'muted' ? 70 : 94,
      l: 45
    }, style),

    neutral: generateColorScale({
      h: 0,
      s: 0, // Achromatic
      l: 50
    }, style)
  };
}
```

### 6. Colorblind Simulation

```javascript
/**
 * Simulate color as perceived by colorblind individuals
 */
function simulateColorblindness(rgb, type) {
  const [r, g, b] = rgb.map(v => v / 255);

  // Transformation matrices for different types
  const matrices = {
    protanopia: [
      [0.56667, 0.43333, 0],
      [0.55833, 0.44167, 0],
      [0, 0.24167, 0.75833]
    ],
    deuteranopia: [
      [0.625, 0.375, 0],
      [0.7, 0.3, 0],
      [0, 0.3, 0.7]
    ],
    tritanopia: [
      [0.95, 0.05, 0],
      [0, 0.43333, 0.56667],
      [0, 0.475, 0.525]
    ]
  };

  const matrix = matrices[type];
  if (!matrix) return rgb;

  const simulated = [
    matrix[0][0] * r + matrix[0][1] * g + matrix[0][2] * b,
    matrix[1][0] * r + matrix[1][1] * g + matrix[1][2] * b,
    matrix[2][0] * r + matrix[2][1] * g + matrix[2][2] * b
  ];

  return simulated.map(v => Math.round(Math.max(0, Math.min(1, v)) * 255));
}

/**
 * Calculate perceptual difference (ΔE) between two colors
 */
function calculateDeltaE(rgb1, rgb2) {
  // Simplified ΔE calculation (CIE76)
  const [r1, g1, b1] = rgb1;
  const [r2, g2, b2] = rgb2;

  const deltaR = r1 - r2;
  const deltaG = g1 - g2;
  const deltaB = b1 - b2;

  return Math.sqrt(deltaR ** 2 + deltaG ** 2 + deltaB ** 2);
}

/**
 * Verify colorblind safety for semantic color pairs
 */
function verifyColorblindSafety(palette) {
  const types = ['protanopia', 'deuteranopia', 'tritanopia'];
  const results = {};

  // Critical pairs to check
  const pairs = [
    ['success-500', 'error-500'],
    ['success-500', 'primary-500'],
    ['warning-500', 'error-500']
  ];

  types.forEach(type => {
    results[type] = {};

    pairs.forEach(([color1, color2]) => {
      const [role1, variant1] = color1.split('-');
      const [role2, variant2] = color2.split('-');

      const rgb1 = hslToRGB(
        palette[role1][variant1].h,
        palette[role1][variant1].s,
        palette[role1][variant1].l
      );
      const rgb2 = hslToRGB(
        palette[role2][variant2].h,
        palette[role2][variant2].s,
        palette[role2][variant2].l
      );

      const sim1 = simulateColorblindness(rgb1, type);
      const sim2 = simulateColorblindness(rgb2, type);

      const deltaE = calculateDeltaE(sim1, sim2);

      results[type][`${color1}-${color2}`] = {
        deltaE,
        safe: deltaE >= 30 // Threshold for distinguishability
      };
    });
  });

  return results;
}
```

### 7. Dark Mode Generation

```javascript
/**
 * Generate dark mode variant of color scale
 */
function generateDarkMode(lightScale, style) {
  const darkScale = {};

  // Invert lightness values
  const inversion = {
    50: 950,
    100: 900,
    200: 800,
    300: 700,
    400: 600,
    500: 500, // Base stays similar but slightly adjusted
    600: 400,
    700: 300,
    800: 200,
    900: 100,
    950: 50
  };

  Object.keys(lightScale).forEach(variant => {
    const original = lightScale[variant];
    const targetVariant = inversion[variant];
    const targetLightness = lightScale[targetVariant]?.l || original.l;

    darkScale[variant] = {
      h: original.h,
      s: Math.max(5, original.s - 5), // Reduce saturation slightly for comfort
      l: targetLightness
    };
  });

  return darkScale;
}
```

## Output Templates

### CSS Custom Properties Template

```css
:root {
  /* Primary Scale */
  --color-primary-50: hsl({h}, {s}%, {l}%);
  --color-primary-100: hsl({h}, {s}%, {l}%);
  --color-primary-200: hsl({h}, {s}%, {l}%);
  --color-primary-300: hsl({h}, {s}%, {l}%);
  --color-primary-400: hsl({h}, {s}%, {l}%);
  --color-primary-500: hsl({h}, {s}%, {l}%);
  --color-primary-600: hsl({h}, {s}%, {l}%);
  --color-primary-700: hsl({h}, {s}%, {l}%);
  --color-primary-800: hsl({h}, {s}%, {l}%);
  --color-primary-900: hsl({h}, {s}%, {l}%);
  --color-primary-950: hsl({h}, {s}%, {l}%);

  /* Text colors for primary backgrounds */
  --color-on-primary-50: hsl(0, 0%, 10%);
  --color-on-primary-100: hsl(0, 0%, 10%);
  --color-on-primary-200: hsl(0, 0%, 10%);
  --color-on-primary-300: hsl(0, 0%, 10%);
  --color-on-primary-400: hsl(0, 0%, 100%);
  --color-on-primary-500: hsl(0, 0%, 100%);
  --color-on-primary-600: hsl(0, 0%, 100%);
  --color-on-primary-700: hsl(0, 0%, 100%);
  --color-on-primary-800: hsl(0, 0%, 100%);
  --color-on-primary-900: hsl(0, 0%, 100%);
  --color-on-primary-950: hsl(0, 0%, 100%);

  /* Success Scale */
  --color-success-50: hsl({h}, {s}%, {l}%);
  --color-success-500: hsl({h}, {s}%, {l}%);
  --color-success-on-500: hsl(0, 0%, 100%);

  /* Warning Scale */
  --color-warning-50: hsl({h}, {s}%, {l}%);
  --color-warning-500: hsl({h}, {s}%, {l}%);
  --color-warning-on-500: hsl(0, 0%, 10%);

  /* Error Scale */
  --color-error-50: hsl({h}, {s}%, {l}%);
  --color-error-500: hsl({h}, {s}%, {l}%);
  --color-error-on-500: hsl(0, 0%, 100%);

  /* Info Scale */
  --color-info-50: hsl({h}, {s}%, {l}%);
  --color-info-500: hsl({h}, {s}%, {l}%);
  --color-info-on-500: hsl(0, 0%, 100%);

  /* Neutral Scale */
  --color-neutral-50: hsl(0, 0%, 98%);
  --color-neutral-100: hsl(0, 0%, 96%);
  --color-neutral-200: hsl(0, 0%, 90%);
  --color-neutral-300: hsl(0, 0%, 80%);
  --color-neutral-400: hsl(0, 0%, 60%);
  --color-neutral-500: hsl(0, 0%, 50%);
  --color-neutral-600: hsl(0, 0%, 40%);
  --color-neutral-700: hsl(0, 0%, 30%);
  --color-neutral-800: hsl(0, 0%, 20%);
  --color-neutral-900: hsl(0, 0%, 15%);
  --color-neutral-950: hsl(0, 0%, 10%);
}

/* Dark Mode */
[data-theme="dark"] {
  --color-primary-50: hsl({h}, {s}%, {l}%);
  /* ... inverted scale ... */
}
```

### Accessibility Report Template

```
╔══════════════════════════════════════════════════════════════════╗
║  ACCESSIBLE COLOR PALETTE REPORT                                 ║
╚══════════════════════════════════════════════════════════════════╝

Base Color: {hex} | hsl({h}, {s}%, {l}%)
Style: {style}
WCAG Level: {level}
Colorblind Safe: {enabled}

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

PRIMARY-500: hsl({h}, {s}%, {l}%) | {hex}
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

Contrast Ratios:
  vs White (#FFFFFF): {ratio}:1
    {✓/✗} WCAG AA Normal Text (4.5:1)
    {✓/✗} WCAG AAA Normal Text (7:1)
    {✓/✗} WCAG AA Large Text (3:1)
    {✓/✗} WCAG AAA Large Text (4.5:1)
    APCA: Lc {score} ({rating})

  vs Black (#000000): {ratio}:1
    {✓/✗} WCAG AA Normal Text (4.5:1)

  vs Neutral-50 ({hex}): {ratio}:1
    {✓/✗} WCAG AA Normal Text

Recommended Text Color: {color}
  Contrast: {ratio}:1 {✓/✗} {level}

Colorblind Safety:
  {✓/✗} Protanopia: {status}
  {✓/✗} Deuteranopia: {status}
  {✓/✗} Tritanopia: {status}

Usage Recommendations:
  {✓/✗} {recommendation 1}
  {✓/✗} {recommendation 2}
  {⚠/✗} {warning or restriction}

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
[Repeat for other key colors: success-500, warning-500, error-500]
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

╔══════════════════════════════════════════════════════════════════╗
║  USAGE EXAMPLES                                                  ║
╚══════════════════════════════════════════════════════════════════╝

1. Primary Button:
   background: var(--color-primary-600);
   color: var(--color-on-primary-600);
   Contrast: {ratio}:1 ✓

2. Success Alert:
   background: var(--color-success-50);
   color: var(--color-success-800);
   Contrast: {ratio}:1 ✓ AAA

3. Error Text:
   color: var(--color-error-700);
   (on white background)
   Contrast: {ratio}:1 ✓ AAA

╔══════════════════════════════════════════════════════════════════╗
║  WARNINGS & RECOMMENDATIONS                                      ║
╚══════════════════════════════════════════════════════════════════╝

⚠️ {specific warning about problematic combinations}
✓ {best practice suggestion}
💡 {optimization tip}
```

## Example Interactions

### Example 1: Basic Palette Generation

**User Input:**
```
Generate an accessible palette from #3B82F6
```

**AI Response:**
```
╔══════════════════════════════════════════════════════════════════╗
║  ACCESSIBLE COLOR PALETTE                                        ║
╚══════════════════════════════════════════════════════════════════╝

Base Color: #3B82F6 | hsl(217, 91%, 60%)
Style: Vibrant
WCAG Level: AA
Colorblind Safe: ✓ Enabled

━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━
PRIMARY SCALE
━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━

50:  hsl(217, 91%, 95%) | #EFF6FF
100: hsl(217, 91%, 90%) | #DBEAFE
200: hsl(217, 91%, 80%) | #BFDBFE
300: hsl(217, 91%, 70%) | #93C5FD
400: hsl(217, 91%, 60%) | #60A5FA
500: hsl(217, 91%, 50%) | #3B82F6
600: hsl(217, 91%, 40%) | #2563EB
700: hsl(217, 91%, 30%) | #1D4ED8
800: hsl(217, 91%, 20%) | #1E40AF
900: hsl(217, 91%, 15%) | #1E3A8A
950: hsl(217, 91%, 10%) | #172554

[Continues with SUCCESS, WARNING, ERROR, INFO, NEUTRAL scales...]

[Includes full accessibility report as shown in template above]

[Includes CSS output, Tailwind config, usage examples...]
```

### Example 2: Advanced Request with Options

**User Input:**
```
Create a WCAG AAA compliant palette from #10B981 with a muted style,
colorblind-safe enabled, export in Tailwind format
```

**AI Response:**
[Complete palette with AAA validation, muted saturation adjustments, colorblind verification, and Tailwind config format]

### Example 3: Accessibility Audit

**User Input:**
```
Check if these colors meet WCAG AA:
Primary: #FF6B6B
Success: #51CF66
Background: #FFFFFF
```

**AI Response:**
```
╔══════════════════════════════════════════════════════════════════╗
║  ACCESSIBILITY AUDIT                                             ║
╚══════════════════════════════════════════════════════════════════╝

PRIMARY (#FF6B6B) on WHITE (#FFFFFF):
  Contrast: 3.2:1
  ✗ WCAG AA Normal Text (requires 4.5:1)
  ✓ WCAG AA Large Text (requires 3:1)
  ✗ WCAG AA UI Components (requires 3:1... barely fails)

  ⚠️ ISSUE: Insufficient contrast for normal text

  ✓ SOLUTION: Darken to #E63946 (contrast: 4.5:1)
  or use for large headings only (18pt+)

SUCCESS (#51CF66) on WHITE (#FFFFFF):
  Contrast: 2.9:1
  ✗ WCAG AA Normal Text (requires 4.5:1)
  ✗ WCAG AA Large Text (requires 3:1... close!)

  ⚠️ ISSUE: Fails all WCAG AA text requirements

  ✓ SOLUTION: Darken to #22C55E (contrast: 4.6:1)

COLORBLIND SAFETY:
  Primary (#FF6B6B) vs Success (#51CF66)
  ✓ Protanopia: ΔE = 42 (safe)
  ⚠️ Deuteranopia: ΔE = 28 (marginally distinguishable)
  ✓ Tritanopia: ΔE = 65 (very safe)

  RECOMMENDATION: Increase lightness difference between
  primary and success to improve deuteranopia safety.
  Suggested: Primary at L=45%, Success at L=55%
```

## Testing Checklist

Before deploying a generated palette, verify:

- [ ] All 11 variants generated for primary, secondary, neutral
- [ ] All semantic colors (success, warning, error, info) generated
- [ ] Every color has calculated contrast ratio vs white and black
- [ ] "on-{color}" text colors provided for all key variants
- [ ] WCAG compliance checked for specified level (AA or AAA)
- [ ] Dark mode variants generated if requested
- [ ] Colorblind safety verified if enabled
- [ ] At least 3 usage examples provided
- [ ] Any failures clearly flagged with alternatives
- [ ] Requested output format(s) included
- [ ] Code is copy-paste ready (no placeholders)

## Common Edge Cases

### Very Light Input Color (L > 90%)
```
⚠️ Your input color (#F8F8F8, L=97%) is too light to create
accessible text on white backgrounds.

SUGGESTIONS:
1. Use this color for backgrounds instead of primary
2. Darken base to at least L=60% for primary actions
3. Recommended primary: hsl(217, 91%, 50%) | #3B82F6
```

### Very Dark Input Color (L < 15%)
```
⚠️ Your input color (#0A0A0A, L=4%) is too dark to generate
accessible light variants (50-300 range will be nearly identical).

SUGGESTIONS:
1. Use this color as neutral-950 instead of primary
2. Lighten base to at least L=40% for primary actions
3. Recommended primary: hsl(217, 91%, 50%) | #3B82F6
```

### Highly Saturated Input (S > 95%)
```
✓ Your color is highly saturated (S=98%). Generated palette
maintains vibrant appearance while ensuring accessibility.

NOTE: Some lighter variants (50-200) have reduced saturation
to prevent visual discomfort on large surfaces.
```

### Colorblind Conflict Detected
```
⚠️ COLORBLIND SAFETY ISSUE:

Your success (#22C55E) and error (#EF4444) colors have a
ΔE of only 25 in deuteranopia simulation (threshold: 30).

This means users with green color blindness may struggle to
distinguish between success and error states.

SOLUTIONS:
1. Increase lightness difference:
   - Success: L=55% (lighter)
   - Error: L=45% (darker)

2. Use additional visual indicators:
   - Icons (✓ for success, ✗ for error)
   - Position (success top, error bottom)
   - Patterns or textures

3. Alternative colors:
   - Success: hsl(142, 71%, 55%) | #34D399 (blue-green)
   - Error: hsl(0, 84%, 45%) | #DC2626 (darker red)
   - ΔE in deuteranopia: 38 ✓
```

## Skill Variables

| Variable | Type | Default | Description |
|----------|------|---------|-------------|
| `primary_color` | String (hex/rgb/hsl) | `#3B82F6` | Base brand color |
| `wcag_level` | String | `AA` | WCAG compliance level: `AA` or `AAA` |
| `style_preference` | String | `vibrant` | Style: `vibrant`, `muted`, `earthy`, `pastel`, `bold`, `minimal` |
| `colorblind_safe` | Boolean | `true` | Enable colorblind verification |
| `output_format` | String | `all` | Export format: `all`, `css`, `tailwind`, `scss`, `json` |
| `dark_mode` | Boolean | `true` | Generate dark mode variants |
| `include_opacity` | Boolean | `false` | Include opacity variants (10%, 20%, 50%, 80%) |
| `include_states` | Boolean | `false` | Include state variants (hover, active, focus, disabled) |

## Integration Examples

### With Tailwind CSS

```javascript
// tailwind.config.js
const palette = require('./palette.json');

module.exports = {
  theme: {
    extend: {
      colors: palette.colors,
    },
  },
};
```

### With CSS-in-JS (Styled Components)

```javascript
// theme.js
export const theme = {
  colors: {
    primary: {
      50: 'hsl(217, 91%, 95%)',
      500: 'hsl(217, 91%, 50%)',
      // ...
    },
  },
};

// component.jsx
import styled from 'styled-components';

const Button = styled.button`
  background: ${props => props.theme.colors.primary[500]};
  color: white;

  &:hover {
    background: ${props => props.theme.colors.primary[600]};
  }
`;
```

### With Figma Tokens Studio

```json
{
  "color": {
    "primary": {
      "500": {
        "value": "{color.primary.500}",
        "type": "color",
        "description": "Primary brand color"
      }
    }
  }
}
```

## References & Resources

- **WCAG 2.1 Guidelines**: https://www.w3.org/WAI/WCAG21/quickref/
- **APCA Contrast Calculator**: https://github.com/Myndex/SAPC-APCA
- **Colorblind Simulator**: https://www.color-blindness.com/coblis-color-blindness-simulator/
- **Contrast Checker**: https://webaim.org/resources/contrastchecker/
- **HSL Color Picker**: https://hslpicker.com/

## Version History

- **v1.0.0** (2025-08-15): Initial release
  - 11-step color scales (50-950)
  - WCAG 2.1 AA/AAA compliance
  - Colorblind-safe verification
  - Dark mode generation
  - Multi-format export (CSS, Tailwind, SCSS, JSON)

## License

This skill is released under MIT License and can be freely used, modified, and distributed.

## Support

For issues, suggestions, or contributions:
- GitHub: [ai-skills-directory](https://github.com/yourusername/ai-skills-directory)
- Email: support@aiskills.directory

---

**This INIT.md file is self-contained and includes all algorithms, templates, and examples needed to recreate the Accessible Color Palette Generator skill.**
