---
name: swing-trading-strategy
version: 1.0.0
description: Swing trading analysis with technical indicators, risk management, and position sizing
author: FindSkill.ai
license: MIT
---

# Initialization

This skill provides a comprehensive swing trading analysis framework. It is primarily prompt-based but includes optional Python scripts for position sizing calculations and trade journaling.

**Time to initialize**: ~3 minutes

## Directory Structure

```
swing-trading-strategy/
├── SKILL.md                    # Main skill instructions
├── INIT.md                     # This initialization file
├── scripts/
│   ├── position_calculator.py  # Position sizing calculator
│   └── trade_journal.py        # Trade journal manager
└── references/
    └── indicator_cheatsheet.md # Quick reference for indicators
```

## Dependencies

```bash
pip install pandas tabulate
```

## Scripts

### scripts/position_calculator.py
**Purpose**: Calculate position size based on risk parameters
**Usage**: `python scripts/position_calculator.py --account 10000 --risk 2 --entry 50 --stop 47`

```python
#!/usr/bin/env python3
"""
Swing Trading Position Size Calculator
Calculate position size based on account risk and stop loss distance.
"""

import argparse
from typing import Tuple

def calculate_position_size(
    account_balance: float,
    risk_percent: float,
    entry_price: float,
    stop_loss: float
) -> Tuple[int, float, float, float]:
    """
    Calculate position size for a swing trade.

    Args:
        account_balance: Total trading capital
        risk_percent: Percentage of account to risk (e.g., 2 for 2%)
        entry_price: Planned entry price
        stop_loss: Stop loss price

    Returns:
        Tuple of (shares, position_value, dollar_risk, risk_per_share)
    """
    # Calculate dollar amount at risk
    dollar_risk = account_balance * (risk_percent / 100)

    # Calculate risk per share
    risk_per_share = abs(entry_price - stop_loss)

    if risk_per_share == 0:
        raise ValueError("Entry price and stop loss cannot be the same")

    # Calculate number of shares
    shares = int(dollar_risk / risk_per_share)

    # Calculate total position value
    position_value = shares * entry_price

    return shares, position_value, dollar_risk, risk_per_share


def calculate_targets(
    entry_price: float,
    stop_loss: float,
    risk_reward_ratios: list = [1.5, 2.0, 3.0]
) -> list:
    """
    Calculate profit targets based on risk:reward ratios.

    Args:
        entry_price: Entry price
        stop_loss: Stop loss price
        risk_reward_ratios: List of R:R ratios for targets

    Returns:
        List of target prices
    """
    risk = abs(entry_price - stop_loss)
    direction = 1 if entry_price > stop_loss else -1

    targets = []
    for rr in risk_reward_ratios:
        target = entry_price + (risk * rr * direction)
        targets.append((rr, round(target, 2)))

    return targets


def print_trade_plan(
    account_balance: float,
    risk_percent: float,
    entry_price: float,
    stop_loss: float,
    ticker: str = "SYMBOL"
):
    """Print a formatted trade plan."""

    shares, position_value, dollar_risk, risk_per_share = calculate_position_size(
        account_balance, risk_percent, entry_price, stop_loss
    )

    targets = calculate_targets(entry_price, stop_loss)

    direction = "LONG" if entry_price > stop_loss else "SHORT"

    print("\n" + "="*50)
    print(f"SWING TRADE PLAN: {ticker} ({direction})")
    print("="*50)

    print(f"\n📊 POSITION DETAILS")
    print(f"   Entry Price:     ${entry_price:.2f}")
    print(f"   Stop Loss:       ${stop_loss:.2f}")
    print(f"   Risk Per Share:  ${risk_per_share:.2f}")

    print(f"\n💰 POSITION SIZING")
    print(f"   Account Balance: ${account_balance:,.2f}")
    print(f"   Risk Percentage: {risk_percent}%")
    print(f"   Dollar Risk:     ${dollar_risk:.2f}")
    print(f"   Shares to Buy:   {shares}")
    print(f"   Position Value:  ${position_value:,.2f}")
    print(f"   % of Account:    {(position_value/account_balance)*100:.1f}%")

    print(f"\n🎯 PROFIT TARGETS")
    for rr, target in targets:
        profit = abs(target - entry_price) * shares
        print(f"   Target {rr}R:      ${target:.2f} (Profit: ${profit:.2f})")

    print(f"\n⚠️  RISK SUMMARY")
    print(f"   Max Loss:        ${dollar_risk:.2f}")
    print(f"   At Target 2R:    ${abs(targets[1][1] - entry_price) * shares:.2f}")
    print(f"   Risk:Reward:     1:{targets[1][0]}")

    print("\n" + "="*50 + "\n")


def main():
    parser = argparse.ArgumentParser(
        description="Calculate swing trade position size"
    )
    parser.add_argument("--account", type=float, required=True,
                        help="Account balance")
    parser.add_argument("--risk", type=float, default=2.0,
                        help="Risk percentage (default: 2)")
    parser.add_argument("--entry", type=float, required=True,
                        help="Entry price")
    parser.add_argument("--stop", type=float, required=True,
                        help="Stop loss price")
    parser.add_argument("--ticker", type=str, default="SYMBOL",
                        help="Ticker symbol")

    args = parser.parse_args()

    print_trade_plan(
        args.account,
        args.risk,
        args.entry,
        args.stop,
        args.ticker
    )


if __name__ == "__main__":
    main()
```

### scripts/trade_journal.py
**Purpose**: Track and analyze swing trades
**Usage**: `python scripts/trade_journal.py add --ticker AAPL --entry 150 --exit 165 --shares 50`

```python
#!/usr/bin/env python3
"""
Swing Trading Journal
Track trades and analyze performance.
"""

import argparse
import json
import os
from datetime import datetime
from typing import Optional

JOURNAL_FILE = "trade_journal.json"


def load_journal() -> list:
    """Load existing journal or create new one."""
    if os.path.exists(JOURNAL_FILE):
        with open(JOURNAL_FILE, 'r') as f:
            return json.load(f)
    return []


def save_journal(journal: list):
    """Save journal to file."""
    with open(JOURNAL_FILE, 'w') as f:
        json.dump(journal, f, indent=2, default=str)


def add_trade(
    ticker: str,
    entry_price: float,
    exit_price: float,
    shares: int,
    direction: str = "LONG",
    entry_date: Optional[str] = None,
    exit_date: Optional[str] = None,
    setup_type: str = "",
    notes: str = ""
):
    """Add a completed trade to the journal."""
    journal = load_journal()

    # Calculate P&L
    if direction.upper() == "LONG":
        pnl = (exit_price - entry_price) * shares
    else:
        pnl = (entry_price - exit_price) * shares

    pnl_percent = ((exit_price - entry_price) / entry_price) * 100
    if direction.upper() == "SHORT":
        pnl_percent = -pnl_percent

    trade = {
        "id": len(journal) + 1,
        "ticker": ticker.upper(),
        "direction": direction.upper(),
        "entry_price": entry_price,
        "exit_price": exit_price,
        "shares": shares,
        "position_size": entry_price * shares,
        "pnl": round(pnl, 2),
        "pnl_percent": round(pnl_percent, 2),
        "entry_date": entry_date or datetime.now().strftime("%Y-%m-%d"),
        "exit_date": exit_date or datetime.now().strftime("%Y-%m-%d"),
        "setup_type": setup_type,
        "notes": notes,
        "result": "WIN" if pnl > 0 else "LOSS"
    }

    journal.append(trade)
    save_journal(journal)

    print(f"\n✅ Trade added: {ticker} {direction}")
    print(f"   P&L: ${pnl:.2f} ({pnl_percent:.2f}%)")
    print(f"   Result: {trade['result']}\n")


def show_stats():
    """Display trading statistics."""
    journal = load_journal()

    if not journal:
        print("\n📊 No trades in journal yet.\n")
        return

    total_trades = len(journal)
    wins = sum(1 for t in journal if t['result'] == 'WIN')
    losses = total_trades - wins
    win_rate = (wins / total_trades) * 100 if total_trades > 0 else 0

    total_pnl = sum(t['pnl'] for t in journal)
    avg_win = sum(t['pnl'] for t in journal if t['result'] == 'WIN') / wins if wins > 0 else 0
    avg_loss = sum(t['pnl'] for t in journal if t['result'] == 'LOSS') / losses if losses > 0 else 0

    profit_factor = abs(avg_win * wins) / abs(avg_loss * losses) if losses > 0 and avg_loss != 0 else float('inf')

    print("\n" + "="*50)
    print("SWING TRADING PERFORMANCE")
    print("="*50)

    print(f"\n📈 OVERALL STATS")
    print(f"   Total Trades:    {total_trades}")
    print(f"   Wins:            {wins}")
    print(f"   Losses:          {losses}")
    print(f"   Win Rate:        {win_rate:.1f}%")

    print(f"\n💰 P&L ANALYSIS")
    print(f"   Total P&L:       ${total_pnl:,.2f}")
    print(f"   Avg Win:         ${avg_win:,.2f}")
    print(f"   Avg Loss:        ${avg_loss:,.2f}")
    print(f"   Profit Factor:   {profit_factor:.2f}")

    # Recent trades
    print(f"\n📋 RECENT TRADES")
    for trade in journal[-5:]:
        emoji = "🟢" if trade['result'] == 'WIN' else "🔴"
        print(f"   {emoji} {trade['ticker']}: ${trade['pnl']:.2f} ({trade['pnl_percent']:.1f}%)")

    print("\n" + "="*50 + "\n")


def list_trades(limit: int = 10):
    """List recent trades."""
    journal = load_journal()

    if not journal:
        print("\n📋 No trades recorded yet.\n")
        return

    print("\n" + "="*70)
    print("TRADE HISTORY")
    print("="*70)
    print(f"{'ID':<4} {'Ticker':<8} {'Dir':<6} {'Entry':<10} {'Exit':<10} {'P&L':<12} {'Result'}")
    print("-"*70)

    for trade in journal[-limit:]:
        pnl_str = f"${trade['pnl']:.2f}"
        print(f"{trade['id']:<4} {trade['ticker']:<8} {trade['direction']:<6} "
              f"${trade['entry_price']:<9.2f} ${trade['exit_price']:<9.2f} "
              f"{pnl_str:<12} {trade['result']}")

    print("="*70 + "\n")


def main():
    parser = argparse.ArgumentParser(description="Swing Trade Journal")
    subparsers = parser.add_subparsers(dest="command", help="Commands")

    # Add trade
    add_parser = subparsers.add_parser("add", help="Add a trade")
    add_parser.add_argument("--ticker", required=True, help="Ticker symbol")
    add_parser.add_argument("--entry", type=float, required=True, help="Entry price")
    add_parser.add_argument("--exit", type=float, required=True, help="Exit price")
    add_parser.add_argument("--shares", type=int, required=True, help="Number of shares")
    add_parser.add_argument("--direction", default="LONG", help="LONG or SHORT")
    add_parser.add_argument("--setup", default="", help="Setup type")
    add_parser.add_argument("--notes", default="", help="Trade notes")

    # Stats
    subparsers.add_parser("stats", help="Show trading statistics")

    # List
    list_parser = subparsers.add_parser("list", help="List trades")
    list_parser.add_argument("--limit", type=int, default=10, help="Number to show")

    args = parser.parse_args()

    if args.command == "add":
        add_trade(
            args.ticker, args.entry, args.exit, args.shares,
            args.direction, setup_type=args.setup, notes=args.notes
        )
    elif args.command == "stats":
        show_stats()
    elif args.command == "list":
        list_trades(args.limit)
    else:
        parser.print_help()


if __name__ == "__main__":
    main()
```

## References

### references/indicator_cheatsheet.md

```markdown
# Technical Indicator Cheatsheet

Quick reference for swing trading indicators.

## RSI (Relative Strength Index)

| Level | Signal | Action |
|-------|--------|--------|
| > 70 | Overbought | Consider taking profits / shorting |
| 50-70 | Bullish | Hold longs, look for pullback entries |
| 30-50 | Neutral | Watch for direction confirmation |
| < 30 | Oversold | Look for long entries |

**RSI Divergence:**
- Bullish: Price makes lower low, RSI makes higher low → Buy signal
- Bearish: Price makes higher high, RSI makes lower high → Sell signal

## Moving Averages

| MA | Period | Use Case |
|----|--------|----------|
| 10 EMA | Short-term | Aggressive momentum trades |
| 20 EMA | Short-term | Standard swing trading entries |
| 50 SMA | Medium-term | Trend confirmation, pullback support |
| 200 SMA | Long-term | Major trend direction, key S/R |

**Golden Cross:** 50 MA crosses above 200 MA → Bullish
**Death Cross:** 50 MA crosses below 200 MA → Bearish

## MACD

| Signal | Meaning | Action |
|--------|---------|--------|
| MACD > Signal | Bullish momentum | Look for longs |
| MACD < Signal | Bearish momentum | Look for shorts |
| Histogram expanding | Momentum increasing | Hold position |
| Histogram contracting | Momentum weakening | Tighten stops |

## Bollinger Bands (20,2)

| Location | Signal |
|----------|--------|
| Upper band touch | Overbought, potential pullback |
| Lower band touch | Oversold, potential bounce |
| Band squeeze | Low volatility, breakout pending |
| Band expansion | High volatility, trend move |

## Volume

| Pattern | Meaning |
|---------|---------|
| High volume + price up | Strong bullish move |
| High volume + price down | Strong bearish move |
| Low volume + price up | Weak rally, potential reversal |
| Volume spike at support | Potential reversal |

## Key Candlestick Patterns

**Bullish:**
- Hammer (at support)
- Bullish engulfing
- Morning star
- Doji at support

**Bearish:**
- Shooting star (at resistance)
- Bearish engulfing
- Evening star
- Doji at resistance
```

## File Extraction Rules

1. **Script files**: Extract code block after `### scripts/{filename}` → save to `scripts/`
2. **Reference files**: Extract code block after `### references/{filename}` → save to `references/`
3. Create directories if they don't exist
4. Make Python scripts executable: `chmod +x scripts/*.py`

## Post-Init Steps

1. Create the directory structure:
   ```bash
   mkdir -p scripts references
   ```

2. Extract and save each file from the code blocks above

3. Install dependencies:
   ```bash
   pip install pandas tabulate
   ```

4. Verify installation:
   ```bash
   python scripts/position_calculator.py --account 10000 --risk 2 --entry 50 --stop 47 --ticker TEST
   ```

5. Test trade journal:
   ```bash
   python scripts/trade_journal.py add --ticker TEST --entry 50 --exit 55 --shares 100
   python scripts/trade_journal.py stats
   ```

## Compatibility

Tested with: Claude, ChatGPT, Gemini, Copilot

The main skill (SKILL.md) works without initialization. Scripts enhance the experience but are optional.

---
Downloaded from [Find Skill.ai](https://findskill.ai)
