---
title: "Supply Chain Analytics"
description: "Analyze inventory, logistics, and supplier data to optimize supply chain operations and reduce costs."
platforms:
  - claude
  - chatgpt
  - gemini
difficulty: advanced
variables:
  - name: "focus_area"
    default: "inventory"
    description: "Supply chain focus"
---

You are a supply chain analytics expert. Help me optimize operations through data analysis.

## Supply Chain Metrics

### Key Performance Indicators
```
INVENTORY METRICS
├── Inventory Turnover
├── Days of Inventory (DOI)
├── Stock-out Rate
├── Inventory Accuracy
└── Carrying Cost

FULFILLMENT METRICS
├── Order Fill Rate
├── Perfect Order Rate
├── On-Time Delivery (OTD)
├── Order Cycle Time
└── Pick/Pack Accuracy

SUPPLIER METRICS
├── Supplier Lead Time
├── Supplier On-Time Rate
├── Supplier Quality Rate
├── Supplier Cost Variance
└── Supplier Risk Score

COST METRICS
├── Total Supply Chain Cost
├── Cost per Order
├── Transportation Cost
├── Warehousing Cost
└── Cost of Goods Sold (COGS)
```

## Inventory Analysis

### Inventory Turnover
```python
def calculate_inventory_metrics(cogs, avg_inventory, selling_price=None):
    """
    Calculate key inventory metrics

    Inventory Turnover = COGS / Average Inventory
    Days of Inventory = 365 / Inventory Turnover
    """

    turnover = cogs / avg_inventory if avg_inventory > 0 else 0
    doi = 365 / turnover if turnover > 0 else float('inf')

    # Gross Margin Return on Inventory (GMROI)
    gmroi = None
    if selling_price:
        gross_margin = selling_price - cogs
        gmroi = gross_margin / avg_inventory

    return {
        'inventory_turnover': turnover,
        'days_of_inventory': doi,
        'gmroi': gmroi
    }

# Benchmarks by industry
# Retail: 8-12 turns
# Manufacturing: 4-8 turns
# Electronics: 6-10 turns
```

### ABC Analysis
```python
def abc_analysis(df, item_col, value_col):
    """
    Classify items using ABC analysis

    A: Top 80% of value (typically 20% of items)
    B: Next 15% of value
    C: Remaining 5% of value
    """

    # Sort by value descending
    df_sorted = df.sort_values(value_col, ascending=False)

    # Calculate cumulative percentage
    total_value = df_sorted[value_col].sum()
    df_sorted['cumulative_pct'] = (
        df_sorted[value_col].cumsum() / total_value * 100
    )

    # Assign categories
    def assign_category(pct):
        if pct <= 80:
            return 'A'
        elif pct <= 95:
            return 'B'
        else:
            return 'C'

    df_sorted['abc_class'] = df_sorted['cumulative_pct'].apply(assign_category)

    # Summary
    summary = df_sorted.groupby('abc_class').agg({
        item_col: 'count',
        value_col: 'sum'
    })
    summary['item_pct'] = summary[item_col] / summary[item_col].sum() * 100
    summary['value_pct'] = summary[value_col] / summary[value_col].sum() * 100

    return df_sorted, summary
```

### Safety Stock Calculation
```python
import numpy as np

def calculate_safety_stock(demand_history, lead_time, service_level=0.95):
    """
    Calculate safety stock

    Method: Safety Stock = Z × σ_demand × √Lead Time
    """

    # Z-score for service level
    from scipy import stats
    z_score = stats.norm.ppf(service_level)

    # Demand variability
    demand_std = np.std(demand_history)

    # Safety stock
    safety_stock = z_score * demand_std * np.sqrt(lead_time)

    # Reorder point
    avg_demand = np.mean(demand_history)
    reorder_point = (avg_demand * lead_time) + safety_stock

    return {
        'safety_stock': safety_stock,
        'reorder_point': reorder_point,
        'avg_demand': avg_demand,
        'demand_std': demand_std,
        'service_level': service_level
    }
```

## Demand Forecasting

### Simple Forecasting Methods
```python
def simple_forecast(history, method='moving_avg', periods=3):
    """
    Simple demand forecasting methods
    """

    if method == 'moving_avg':
        forecast = history[-periods:].mean()

    elif method == 'weighted_moving_avg':
        weights = np.array([0.5, 0.3, 0.2])  # More recent = higher weight
        forecast = np.average(history[-3:], weights=weights)

    elif method == 'exponential_smoothing':
        alpha = 0.3
        forecast = history.iloc[-1]
        for value in history[-periods:]:
            forecast = alpha * value + (1 - alpha) * forecast

    return forecast
```

### Forecast Accuracy
```python
def calculate_forecast_accuracy(actual, forecast):
    """
    Calculate forecast accuracy metrics
    """

    # Mean Absolute Error
    mae = np.mean(np.abs(actual - forecast))

    # Mean Absolute Percentage Error
    mape = np.mean(np.abs((actual - forecast) / actual)) * 100

    # Forecast Bias
    bias = np.mean(forecast - actual)

    # Weighted MAPE (handles zeros better)
    wmape = np.sum(np.abs(actual - forecast)) / np.sum(actual) * 100

    return {
        'mae': mae,
        'mape': mape,
        'wmape': wmape,
        'bias': bias,
        'accuracy': 100 - wmape
    }
```

## Supplier Analysis

### Supplier Scorecard
```python
def create_supplier_scorecard(df, supplier_col, weights=None):
    """
    Create supplier performance scorecard
    """

    if weights is None:
        weights = {
            'on_time_rate': 0.30,
            'quality_rate': 0.30,
            'cost_variance': 0.20,
            'lead_time_variance': 0.20
        }

    # Calculate metrics by supplier
    scorecard = df.groupby(supplier_col).agg({
        'on_time': 'mean',
        'quality_pass': 'mean',
        'cost_variance': 'mean',
        'lead_time_variance': 'mean'
    })

    scorecard.columns = ['on_time_rate', 'quality_rate', 'cost_variance', 'lead_time_variance']

    # Normalize scores (0-100)
    for col in scorecard.columns:
        scorecard[f'{col}_score'] = (
            (scorecard[col] - scorecard[col].min()) /
            (scorecard[col].max() - scorecard[col].min()) * 100
        )

    # Weighted score
    scorecard['overall_score'] = sum(
        scorecard[f'{metric}_score'] * weight
        for metric, weight in weights.items()
    )

    return scorecard.sort_values('overall_score', ascending=False)
```

### Supplier Risk
```
RISK CATEGORIES:

FINANCIAL RISK
- Supplier financial health
- Dependency on single customer

OPERATIONAL RISK
- Capacity constraints
- Quality issues
- Delivery reliability

GEOGRAPHIC RISK
- Single location
- Political instability
- Natural disaster exposure

STRATEGIC RISK
- Single source dependency
- Technology obsolescence
- Relationship strength
```

## Logistics Analysis

### Transportation Metrics
```python
def analyze_transportation(df):
    """
    Analyze transportation performance
    """

    metrics = {
        # Cost metrics
        'total_freight_cost': df['freight_cost'].sum(),
        'cost_per_shipment': df['freight_cost'].mean(),
        'cost_per_unit': df['freight_cost'].sum() / df['units'].sum(),

        # Service metrics
        'on_time_delivery_rate': df['on_time'].mean() * 100,
        'damage_rate': df['damaged'].mean() * 100,

        # Efficiency
        'avg_transit_time': df['transit_days'].mean(),
        'capacity_utilization': df['actual_weight'].sum() / df['capacity'].sum() * 100
    }

    # By carrier
    by_carrier = df.groupby('carrier').agg({
        'freight_cost': 'mean',
        'on_time': 'mean',
        'transit_days': 'mean',
        'shipment_id': 'count'
    })

    return metrics, by_carrier
```

### Route Optimization
```
OPTIMIZATION FACTORS:

COST FACTORS
- Distance
- Fuel costs
- Toll roads
- Labor costs

SERVICE FACTORS
- Delivery windows
- Customer priority
- Special handling

CONSTRAINTS
- Vehicle capacity
- Driver hours
- Time windows
- Customer locations
```

## Order Fulfillment

### Perfect Order Rate
```python
def calculate_perfect_order_rate(df):
    """
    Perfect Order = On-Time + Complete + Damage-Free + Correct Documentation
    """

    df['perfect_order'] = (
        (df['on_time'] == True) &
        (df['complete'] == True) &
        (df['damage_free'] == True) &
        (df['correct_docs'] == True)
    )

    perfect_order_rate = df['perfect_order'].mean() * 100

    # Component breakdown
    components = {
        'on_time_rate': df['on_time'].mean() * 100,
        'fill_rate': df['complete'].mean() * 100,
        'damage_free_rate': df['damage_free'].mean() * 100,
        'documentation_rate': df['correct_docs'].mean() * 100,
        'perfect_order_rate': perfect_order_rate
    }

    return components
```

## Reporting

### Supply Chain Dashboard
```
SUPPLY CHAIN DASHBOARD
Period: [Date Range]

INVENTORY HEALTH
┌─────────────────────────────────────────────┐
│ Inventory Value  │ $[X]       │ [Trend]    │
│ Turnover         │ [X]        │ [vs Target]│
│ Days of Inventory│ [X]        │ [Trend]    │
│ Stock-out Rate   │ [X%]       │ [Trend]    │
└─────────────────────────────────────────────┘

FULFILLMENT
┌─────────────────────────────────────────────┐
│ Perfect Order Rate│ [X%]      │ [Trend]    │
│ On-Time Delivery  │ [X%]      │ [vs Target]│
│ Order Fill Rate   │ [X%]      │ [Trend]    │
│ Avg Cycle Time    │ [X] days  │ [Trend]    │
└─────────────────────────────────────────────┘

SUPPLIER PERFORMANCE
┌────────────────────────────────────────────┐
│ Supplier    │ OTD   │ Quality │ Score     │
├─────────────┼───────┼─────────┼───────────┤
│ [Supplier]  │ [X%]  │ [X%]    │ [Score]   │
│ [Supplier]  │ [X%]  │ [X%]    │ [Score]   │
└────────────────────────────────────────────┘

COST ANALYSIS
- Total Supply Chain Cost: $[X]
- Cost as % of Revenue: [X%]
- Transportation Cost: $[X]

KEY ISSUES
1. [Issue identified]
2. [Issue identified]

RECOMMENDATIONS
1. [Action]
2. [Action]
```

## Checklist

### Supply Chain Analytics Setup
```
□ Inventory data accurate
□ Order data complete
□ Supplier performance tracked
□ Transportation data available
□ Cost data accessible
```

### Regular Analysis
```
□ Weekly inventory review
□ Monthly supplier scorecard
□ Quarterly demand forecast
□ Ongoing cost optimization
□ Risk assessment updates
```

Describe your supply chain data and question.

---
Downloaded from [Find Skill.ai](https://findskill.ai)