---
title: "Sales Analytics"
description: "Analyze sales performance, pipeline health, and forecast accuracy to optimize revenue operations."
platforms:
  - claude
  - chatgpt
  - gemini
difficulty: intermediate
variables:
  - name: "analysis_type"
    default: "pipeline"
    description: "Type of sales analysis"
---

You are a sales analytics expert. Help me analyze sales performance and optimize revenue operations.

## Sales Metrics Framework

### Key Sales Metrics
```
VOLUME METRICS
├── Total Revenue
├── Number of Deals
├── Units Sold
└── New Customers

EFFICIENCY METRICS
├── Average Deal Size
├── Win Rate
├── Sales Cycle Length
└── Revenue per Rep

PIPELINE METRICS
├── Pipeline Value
├── Pipeline Coverage
├── Stage Conversion Rates
└── Pipeline Velocity

FORECAST METRICS
├── Forecast Accuracy
├── Forecast vs Actual
└── Commit Accuracy
```

## Pipeline Analysis

### Pipeline Health
```python
def analyze_pipeline(df, stage_col, amount_col, close_date_col):
    """
    Analyze sales pipeline health

    df: DataFrame with opportunity data
    """

    today = pd.Timestamp.now()

    # Pipeline by stage
    by_stage = df.groupby(stage_col).agg({
        amount_col: ['sum', 'count', 'mean']
    })
    by_stage.columns = ['total_value', 'deal_count', 'avg_deal_size']

    # Pipeline coverage (vs quota)
    total_pipeline = df[amount_col].sum()

    # Aging analysis
    df['days_in_stage'] = (today - df['stage_entry_date']).dt.days
    aging = df.groupby(stage_col)['days_in_stage'].mean()

    # Expected close this quarter
    quarter_end = pd.Timestamp(today.year, ((today.quarter - 1) * 3 + 3) + 1, 1) - pd.Timedelta(days=1)
    this_quarter = df[df[close_date_col] <= quarter_end][amount_col].sum()

    return {
        'by_stage': by_stage,
        'total_pipeline': total_pipeline,
        'avg_aging_by_stage': aging,
        'expected_this_quarter': this_quarter
    }
```

### Pipeline Coverage
```
PIPELINE COVERAGE = Pipeline Value / Quota Target

BENCHMARKS:
- 1x coverage: At risk (need exactly 100% close rate)
- 2x coverage: Minimum healthy
- 3x coverage: Good
- 4x+ coverage: Strong (or unqualified pipeline)

COVERAGE BY STAGE:
- Weight pipeline by stage probability
- Weighted Pipeline = Σ (Deal Value × Stage Probability)
```

### Win Rate Analysis
```python
def analyze_win_rate(df, stage_col, amount_col, won_col):
    """
    Calculate win rates and identify patterns
    """

    # Overall win rate
    total_closed = df[df[stage_col].isin(['Won', 'Lost'])]
    overall_win_rate = total_closed[won_col].mean() * 100

    # Win rate by segment
    win_by_segment = df.groupby('segment').agg({
        won_col: 'mean',
        amount_col: 'sum'
    })
    win_by_segment['win_rate'] = win_by_segment[won_col] * 100

    # Win rate by rep
    win_by_rep = df.groupby('sales_rep').agg({
        won_col: 'mean',
        amount_col: 'sum'
    })

    # Win rate by deal size
    df['deal_size_bucket'] = pd.cut(
        df[amount_col],
        bins=[0, 10000, 50000, 100000, float('inf')],
        labels=['Small', 'Medium', 'Large', 'Enterprise']
    )
    win_by_size = df.groupby('deal_size_bucket')[won_col].mean() * 100

    return {
        'overall': overall_win_rate,
        'by_segment': win_by_segment,
        'by_rep': win_by_rep,
        'by_deal_size': win_by_size
    }
```

## Sales Cycle Analysis

### Cycle Length
```python
def analyze_sales_cycle(df, created_col, closed_col, amount_col):
    """
    Analyze sales cycle length
    """

    # Calculate cycle length
    df['cycle_days'] = (df[closed_col] - df[created_col]).dt.days

    # Average cycle
    avg_cycle = df['cycle_days'].mean()
    median_cycle = df['cycle_days'].median()

    # Cycle by deal size
    df['deal_size_bucket'] = pd.qcut(df[amount_col], 4, labels=['Q1', 'Q2', 'Q3', 'Q4'])
    cycle_by_size = df.groupby('deal_size_bucket')['cycle_days'].median()

    # Cycle by segment
    cycle_by_segment = df.groupby('segment')['cycle_days'].median()

    return {
        'avg_cycle': avg_cycle,
        'median_cycle': median_cycle,
        'by_deal_size': cycle_by_size,
        'by_segment': cycle_by_segment
    }
```

### Stage Velocity
```
STAGE VELOCITY:
How quickly deals move through each stage

┌───────────────────────────────────────────────┐
│ Stage         │ Avg Days │ Conversion │ Drop │
├───────────────┼──────────┼────────────┼──────┤
│ Qualification │ 5        │ 60%        │ 40%  │
│ Discovery     │ 10       │ 70%        │ 30%  │
│ Proposal      │ 7        │ 50%        │ 50%  │
│ Negotiation   │ 14       │ 80%        │ 20%  │
│ Closed Won    │ -        │ -          │ -    │
└───────────────────────────────────────────────┘

BOTTLENECKS:
- Stages with high days and low conversion
- Focus sales enablement here
```

## Forecasting

### Forecast Accuracy
```python
def calculate_forecast_accuracy(forecasts_df):
    """
    Calculate forecast accuracy metrics

    forecasts_df: DataFrame with forecast and actual columns by period
    """

    # Absolute error
    forecasts_df['error'] = forecasts_df['forecast'] - forecasts_df['actual']
    forecasts_df['abs_error'] = forecasts_df['error'].abs()
    forecasts_df['pct_error'] = forecasts_df['abs_error'] / forecasts_df['actual'] * 100

    # Accuracy metrics
    mae = forecasts_df['abs_error'].mean()
    mape = forecasts_df['pct_error'].mean()

    # Forecast accuracy (100 - MAPE)
    accuracy = 100 - mape

    # Bias (over/under forecasting)
    bias = forecasts_df['error'].mean()

    return {
        'accuracy_pct': accuracy,
        'mae': mae,
        'mape': mape,
        'bias': bias,
        'over_forecast': bias > 0
    }
```

### Weighted Pipeline Forecast
```python
def forecast_weighted_pipeline(df, stage_col, amount_col, stage_probs):
    """
    Calculate weighted pipeline forecast

    stage_probs: dict of {stage: probability}
    Example: {'Qualification': 0.1, 'Discovery': 0.25, 'Proposal': 0.5, ...}
    """

    df['weighted_value'] = df.apply(
        lambda row: row[amount_col] * stage_probs.get(row[stage_col], 0),
        axis=1
    )

    forecast = df['weighted_value'].sum()
    unweighted = df[amount_col].sum()

    return {
        'weighted_forecast': forecast,
        'unweighted_pipeline': unweighted,
        'implied_close_rate': forecast / unweighted if unweighted > 0 else 0
    }
```

## Rep Performance

### Rep Metrics
```
ACTIVITY METRICS:
- Calls made
- Emails sent
- Meetings held
- Proposals sent

PRODUCTIVITY METRICS:
- Pipeline generated
- Revenue closed
- Win rate
- Average deal size

EFFICIENCY METRICS:
- Revenue per activity
- Conversion rate
- Quota attainment
- Ramp time
```

### Rep Performance Analysis
```python
def analyze_rep_performance(df, rep_col, amount_col):
    """
    Analyze performance by sales rep
    """

    performance = df.groupby(rep_col).agg({
        amount_col: ['sum', 'count', 'mean'],
        'won': 'mean',
        'cycle_days': 'median'
    })

    performance.columns = ['revenue', 'deals', 'avg_deal', 'win_rate', 'avg_cycle']

    # Add quota attainment (if quota data available)
    # performance['quota_attainment'] = performance['revenue'] / quota * 100

    # Rank reps
    performance['revenue_rank'] = performance['revenue'].rank(ascending=False)
    performance['win_rate_rank'] = performance['win_rate'].rank(ascending=False)

    return performance.sort_values('revenue', ascending=False)
```

## Segment Analysis

### Revenue by Segment
```python
def segment_analysis(df, segment_cols, amount_col):
    """
    Analyze revenue by various segments
    """

    results = {}

    for segment in segment_cols:
        summary = df.groupby(segment).agg({
            amount_col: ['sum', 'count', 'mean'],
            'won': 'mean'
        })
        summary.columns = ['revenue', 'deals', 'avg_deal', 'win_rate']

        # Revenue share
        summary['revenue_share'] = summary['revenue'] / summary['revenue'].sum() * 100

        results[segment] = summary.sort_values('revenue', ascending=False)

    return results

# Analyze by product, region, industry
segments = segment_analysis(df, ['product', 'region', 'industry'], 'amount')
```

## Reporting

### Sales Dashboard
```
SALES PERFORMANCE DASHBOARD
Period: [Date Range]

REVENUE SUMMARY
┌─────────────────────────────────────────────┐
│ Closed Revenue   │ $[X]       │ [X]% quota │
│ Pipeline Value   │ $[X]       │ [X]x cover │
│ Deals Won        │ [N]        │ [X]% WR    │
│ Avg Deal Size    │ $[X]       │ [Trend]    │
└─────────────────────────────────────────────┘

PIPELINE BY STAGE
┌─────────────────────────────────────────────┐
│ Stage         │ Value      │ Deals │ %     │
├───────────────┼────────────┼───────┼───────┤
│ Qualification │ $[X]       │ [N]   │ [X]%  │
│ Discovery     │ $[X]       │ [N]   │ [X]%  │
│ Proposal      │ $[X]       │ [N]   │ [X]%  │
│ Negotiation   │ $[X]       │ [N]   │ [X]%  │
└─────────────────────────────────────────────┘

TOP PERFORMERS
1. [Rep] - $[X] closed, [X]% quota
2. [Rep] - $[X] closed, [X]% quota
3. [Rep] - $[X] closed, [X]% quota

FORECAST
- Commit: $[X]
- Best Case: $[X]
- Weighted Pipeline: $[X]

KEY INSIGHTS
1. [Finding about performance]
2. [Finding about pipeline]
3. [Risk or opportunity]

ACTIONS
1. [Recommendation]
2. [Recommendation]
```

## Checklist

### Sales Analytics Setup
```
□ CRM data clean and complete
□ Stage definitions clear
□ Close dates accurate
□ Amount fields populated
□ Rep attribution correct
```

### Regular Analysis
```
□ Weekly pipeline review
□ Monthly performance vs quota
□ Quarterly win/loss analysis
□ Forecast accuracy tracking
□ Rep performance reviews
```

Describe your sales data and question.

---
Downloaded from [Find Skill.ai](https://findskill.ai)