---
name: mean-reversion-strategy
version: 1.0.0
description: Mean reversion trading with Bollinger Bands, Z-scores, and pairs trading
author: FindSkill.ai
license: MIT
---

# Initialization

This skill provides a comprehensive mean reversion trading framework. It includes Python scripts for calculating Bollinger Bands, Z-scores, and analyzing pairs for correlation trading.

**Time to initialize**: ~3 minutes

## Directory Structure

```
mean-reversion-strategy/
├── SKILL.md                      # Main skill instructions
├── INIT.md                       # This initialization file
├── scripts/
│   ├── bollinger_analyzer.py     # Bollinger Band analysis
│   ├── zscore_calculator.py      # Z-score and deviation analysis
│   └── pairs_analyzer.py         # Pairs trading correlation
└── references/
    └── mean_reversion_guide.md   # Quick reference
```

## Dependencies

```bash
pip install pandas numpy yfinance scipy
```

## Scripts

### scripts/bollinger_analyzer.py
**Purpose**: Analyze stocks for Bollinger Band mean reversion setups
**Usage**: `python scripts/bollinger_analyzer.py --ticker AAPL --period 20 --std 2`

```python
#!/usr/bin/env python3
"""
Bollinger Band Mean Reversion Analyzer
Identify mean reversion opportunities using Bollinger Bands.
"""

import argparse
from typing import Dict, List, Tuple
import numpy as np

try:
    import yfinance as yf
    import pandas as pd
    YFINANCE_AVAILABLE = True
except ImportError:
    YFINANCE_AVAILABLE = False


def calculate_bollinger_bands(
    prices: List[float],
    period: int = 20,
    std_dev: float = 2.0
) -> Tuple[List[float], List[float], List[float]]:
    """
    Calculate Bollinger Bands.

    Returns:
        Tuple of (upper_band, middle_band, lower_band)
    """
    prices = np.array(prices)
    middle = pd.Series(prices).rolling(window=period).mean()
    std = pd.Series(prices).rolling(window=period).std()

    upper = middle + (std * std_dev)
    lower = middle - (std * std_dev)

    return upper.tolist(), middle.tolist(), lower.tolist()


def calculate_rsi(prices: List[float], period: int = 14) -> float:
    """Calculate RSI."""
    deltas = np.diff(prices)
    gains = np.where(deltas > 0, deltas, 0)
    losses = np.where(deltas < 0, -deltas, 0)

    avg_gain = np.mean(gains[-period:])
    avg_loss = np.mean(losses[-period:])

    if avg_loss == 0:
        return 100.0

    rs = avg_gain / avg_loss
    return 100 - (100 / (1 + rs))


def analyze_mean_reversion(ticker: str, period: int = 20, std_dev: float = 2.0) -> Dict:
    """
    Analyze a stock for mean reversion opportunity.

    Args:
        ticker: Stock symbol
        period: Bollinger Band period
        std_dev: Standard deviation multiplier

    Returns:
        Analysis dictionary
    """
    if not YFINANCE_AVAILABLE:
        raise ImportError("yfinance required. Run: pip install yfinance pandas")

    stock = yf.Ticker(ticker)
    hist = stock.history(period="3mo")

    if hist.empty or len(hist) < period + 10:
        raise ValueError(f"Insufficient data for {ticker}")

    prices = hist['Close'].tolist()
    current_price = prices[-1]

    # Calculate Bollinger Bands
    upper, middle, lower = calculate_bollinger_bands(prices, period, std_dev)

    current_upper = upper[-1]
    current_middle = middle[-1]
    current_lower = lower[-1]

    # Calculate band width and %B
    band_width = (current_upper - current_lower) / current_middle * 100
    percent_b = (current_price - current_lower) / (current_upper - current_lower) * 100

    # Calculate Z-score
    recent_mean = np.mean(prices[-period:])
    recent_std = np.std(prices[-period:])
    z_score = (current_price - recent_mean) / recent_std if recent_std > 0 else 0

    # Calculate RSI
    rsi = calculate_rsi(prices)

    # Determine position relative to bands
    if current_price <= current_lower:
        band_position = "AT/BELOW LOWER BAND"
        deviation = "OVERSOLD"
    elif current_price >= current_upper:
        band_position = "AT/ABOVE UPPER BAND"
        deviation = "OVERBOUGHT"
    elif current_price < current_middle:
        band_position = "BELOW MIDDLE"
        deviation = "SLIGHTLY OVERSOLD"
    else:
        band_position = "ABOVE MIDDLE"
        deviation = "SLIGHTLY OVERBOUGHT"

    # Mean reversion signal
    signal = "NONE"
    signal_strength = 0

    if current_price <= current_lower and rsi < 30:
        signal = "STRONG BUY (MEAN REVERSION)"
        signal_strength = 5
    elif current_price <= current_lower:
        signal = "BUY (LOWER BAND TOUCH)"
        signal_strength = 4
    elif z_score <= -2:
        signal = "BUY (Z-SCORE EXTREME)"
        signal_strength = 4
    elif current_price >= current_upper and rsi > 70:
        signal = "STRONG SELL (MEAN REVERSION)"
        signal_strength = 5
    elif current_price >= current_upper:
        signal = "SELL (UPPER BAND TOUCH)"
        signal_strength = 4
    elif z_score >= 2:
        signal = "SELL (Z-SCORE EXTREME)"
        signal_strength = 4

    # Calculate targets
    targets = {
        "target_1_middle": round(current_middle, 2),
        "target_2_opposite": round(current_upper if current_price < current_middle else current_lower, 2)
    }

    # Check for band squeeze (low volatility)
    avg_band_width = np.mean([
        (u - l) / m * 100
        for u, m, l in zip(upper[-20:], middle[-20:], lower[-20:])
        if not np.isnan(u) and not np.isnan(l) and m > 0
    ])
    is_squeeze = band_width < avg_band_width * 0.7

    return {
        "ticker": ticker,
        "current_price": round(current_price, 2),
        "upper_band": round(current_upper, 2),
        "middle_band": round(current_middle, 2),
        "lower_band": round(current_lower, 2),
        "band_width": round(band_width, 2),
        "percent_b": round(percent_b, 2),
        "z_score": round(z_score, 2),
        "rsi": round(rsi, 1),
        "band_position": band_position,
        "deviation": deviation,
        "signal": signal,
        "signal_strength": signal_strength,
        "targets": targets,
        "is_squeeze": is_squeeze,
        "warning": "BAND SQUEEZE - Breakout pending, avoid mean reversion" if is_squeeze else None
    }


def print_analysis(analysis: Dict):
    """Print formatted analysis."""
    print("\n" + "="*60)
    print(f"MEAN REVERSION ANALYSIS: {analysis['ticker']}")
    print("="*60)

    print(f"\n📊 PRICE & BANDS")
    print(f"   Current Price:    ${analysis['current_price']}")
    print(f"   Upper Band:       ${analysis['upper_band']}")
    print(f"   Middle Band:      ${analysis['middle_band']}")
    print(f"   Lower Band:       ${analysis['lower_band']}")

    print(f"\n📈 POSITION")
    print(f"   Band Position:    {analysis['band_position']}")
    print(f"   %B:               {analysis['percent_b']}%")
    print(f"   Z-Score:          {analysis['z_score']}")
    print(f"   RSI:              {analysis['rsi']}")

    print(f"\n🎯 SIGNAL")
    print(f"   {analysis['signal']}")
    print(f"   Strength:         {analysis['signal_strength']}/5")

    if analysis['warning']:
        print(f"\n⚠️  WARNING: {analysis['warning']}")

    if analysis['signal_strength'] >= 4:
        print(f"\n💰 TARGETS")
        print(f"   Target 1 (Middle): ${analysis['targets']['target_1_middle']}")
        print(f"   Target 2:          ${analysis['targets']['target_2_opposite']}")

        # Calculate potential profit
        if analysis['current_price'] < analysis['middle_band']:
            profit_t1 = (analysis['middle_band'] - analysis['current_price']) / analysis['current_price'] * 100
            print(f"   Potential T1:      +{profit_t1:.1f}%")

    print("\n" + "="*60 + "\n")


def main():
    parser = argparse.ArgumentParser(description="Bollinger Band Mean Reversion Analyzer")

    parser.add_argument("--ticker", type=str, required=True, help="Stock ticker")
    parser.add_argument("--period", type=int, default=20, help="BB period (default: 20)")
    parser.add_argument("--std", type=float, default=2.0, help="Std dev (default: 2.0)")

    args = parser.parse_args()

    analysis = analyze_mean_reversion(args.ticker, args.period, args.std)
    print_analysis(analysis)


if __name__ == "__main__":
    main()
```

### scripts/zscore_calculator.py
**Purpose**: Calculate Z-scores for mean reversion analysis
**Usage**: `python scripts/zscore_calculator.py --ticker AAPL --lookback 20`

```python
#!/usr/bin/env python3
"""
Z-Score Calculator for Mean Reversion
Calculate statistical deviation from mean.
"""

import argparse
from typing import Dict, List
import numpy as np

try:
    import yfinance as yf
    import pandas as pd
    YFINANCE_AVAILABLE = True
except ImportError:
    YFINANCE_AVAILABLE = False


def calculate_zscore(prices: List[float], lookback: int = 20) -> List[float]:
    """
    Calculate rolling Z-scores.

    Args:
        prices: List of prices
        lookback: Lookback period for mean/std

    Returns:
        List of Z-scores
    """
    prices = np.array(prices)
    z_scores = []

    for i in range(len(prices)):
        if i < lookback:
            z_scores.append(np.nan)
        else:
            window = prices[i-lookback:i]
            mean = np.mean(window)
            std = np.std(window)
            if std > 0:
                z = (prices[i] - mean) / std
            else:
                z = 0
            z_scores.append(z)

    return z_scores


def analyze_zscore(ticker: str, lookback: int = 20) -> Dict:
    """Analyze Z-score for mean reversion."""
    if not YFINANCE_AVAILABLE:
        raise ImportError("yfinance required")

    stock = yf.Ticker(ticker)
    hist = stock.history(period="6mo")

    if hist.empty:
        raise ValueError(f"No data for {ticker}")

    prices = hist['Close'].tolist()
    z_scores = calculate_zscore(prices, lookback)

    current_z = z_scores[-1]
    current_price = prices[-1]

    # Calculate mean and bounds
    recent_mean = np.mean(prices[-lookback:])
    recent_std = np.std(prices[-lookback:])

    # Determine signal
    if current_z <= -2:
        signal = "STRONG BUY"
        description = "Price extremely below mean"
    elif current_z <= -1:
        signal = "BUY"
        description = "Price below mean"
    elif current_z >= 2:
        signal = "STRONG SELL"
        description = "Price extremely above mean"
    elif current_z >= 1:
        signal = "SELL"
        description = "Price above mean"
    else:
        signal = "NEUTRAL"
        description = "Price near mean"

    # Historical extremes
    valid_z = [z for z in z_scores if not np.isnan(z)]
    times_below_minus2 = sum(1 for z in valid_z if z <= -2)
    times_above_plus2 = sum(1 for z in valid_z if z >= 2)

    return {
        "ticker": ticker,
        "current_price": round(current_price, 2),
        "current_zscore": round(current_z, 2),
        "mean": round(recent_mean, 2),
        "std": round(recent_std, 2),
        "minus_2_std": round(recent_mean - 2*recent_std, 2),
        "plus_2_std": round(recent_mean + 2*recent_std, 2),
        "signal": signal,
        "description": description,
        "times_below_minus2": times_below_minus2,
        "times_above_plus2": times_above_plus2,
        "lookback": lookback
    }


def print_zscore_analysis(analysis: Dict):
    """Print Z-score analysis."""
    print("\n" + "="*55)
    print(f"Z-SCORE ANALYSIS: {analysis['ticker']}")
    print("="*55)

    print(f"\n📊 CURRENT STATUS")
    print(f"   Price:            ${analysis['current_price']}")
    print(f"   Z-Score:          {analysis['current_zscore']}")
    print(f"   Mean ({analysis['lookback']}d):      ${analysis['mean']}")
    print(f"   Std Dev:          ${analysis['std']}")

    print(f"\n📏 STATISTICAL BANDS")
    print(f"   +2 Std Dev:       ${analysis['plus_2_std']}")
    print(f"   Mean:             ${analysis['mean']}")
    print(f"   -2 Std Dev:       ${analysis['minus_2_std']}")

    print(f"\n🎯 SIGNAL: {analysis['signal']}")
    print(f"   {analysis['description']}")

    print(f"\n📈 HISTORICAL EXTREMES")
    print(f"   Times below -2σ:  {analysis['times_below_minus2']}")
    print(f"   Times above +2σ:  {analysis['times_above_plus2']}")

    if analysis['current_zscore'] <= -2:
        target_profit = (analysis['mean'] - analysis['current_price']) / analysis['current_price'] * 100
        print(f"\n💰 MEAN REVERSION TARGET")
        print(f"   Target (Mean):    ${analysis['mean']}")
        print(f"   Potential:        +{target_profit:.1f}%")

    print("\n" + "="*55 + "\n")


def main():
    parser = argparse.ArgumentParser(description="Z-Score Calculator")
    parser.add_argument("--ticker", type=str, required=True)
    parser.add_argument("--lookback", type=int, default=20)

    args = parser.parse_args()

    analysis = analyze_zscore(args.ticker, args.lookback)
    print_zscore_analysis(analysis)


if __name__ == "__main__":
    main()
```

### scripts/pairs_analyzer.py
**Purpose**: Analyze stock pairs for correlation trading
**Usage**: `python scripts/pairs_analyzer.py --pair KO,PEP`

```python
#!/usr/bin/env python3
"""
Pairs Trading Analyzer
Find and analyze correlated stock pairs for mean reversion.
"""

import argparse
from typing import Dict, List, Tuple
import numpy as np

try:
    import yfinance as yf
    import pandas as pd
    from scipy import stats
    IMPORTS_AVAILABLE = True
except ImportError:
    IMPORTS_AVAILABLE = False


def calculate_correlation(prices1: List[float], prices2: List[float]) -> float:
    """Calculate Pearson correlation coefficient."""
    return np.corrcoef(prices1, prices2)[0, 1]


def calculate_spread(prices1: List[float], prices2: List[float]) -> List[float]:
    """Calculate price spread between two stocks."""
    return [p1 - p2 for p1, p2 in zip(prices1, prices2)]


def calculate_spread_zscore(spread: List[float], lookback: int = 20) -> List[float]:
    """Calculate Z-score of spread."""
    z_scores = []
    for i in range(len(spread)):
        if i < lookback:
            z_scores.append(np.nan)
        else:
            window = spread[i-lookback:i]
            mean = np.mean(window)
            std = np.std(window)
            if std > 0:
                z = (spread[i] - mean) / std
            else:
                z = 0
            z_scores.append(z)
    return z_scores


def analyze_pair(ticker1: str, ticker2: str, lookback: int = 60) -> Dict:
    """
    Analyze a stock pair for mean reversion trading.

    Args:
        ticker1: First stock ticker
        ticker2: Second stock ticker
        lookback: Lookback period for calculations

    Returns:
        Pair analysis dictionary
    """
    if not IMPORTS_AVAILABLE:
        raise ImportError("Required: pip install yfinance pandas scipy")

    # Fetch data
    stock1 = yf.Ticker(ticker1)
    stock2 = yf.Ticker(ticker2)

    hist1 = stock1.history(period="1y")
    hist2 = stock2.history(period="1y")

    if hist1.empty or hist2.empty:
        raise ValueError("Could not fetch data for one or both tickers")

    # Align dates
    df = pd.DataFrame({
        'price1': hist1['Close'],
        'price2': hist2['Close']
    }).dropna()

    prices1 = df['price1'].tolist()
    prices2 = df['price2'].tolist()

    # Calculate correlation
    correlation = calculate_correlation(prices1, prices2)

    # Calculate spread
    spread = calculate_spread(prices1, prices2)
    spread_z = calculate_spread_zscore(spread, lookback)

    current_spread = spread[-1]
    current_spread_z = spread_z[-1]
    spread_mean = np.mean(spread[-lookback:])
    spread_std = np.std(spread[-lookback:])

    # Determine trading signal
    if current_spread_z <= -2:
        signal = f"BUY {ticker1}, SHORT {ticker2}"
        description = "Spread extremely below mean"
        action = "LONG_SPREAD"
    elif current_spread_z >= 2:
        signal = f"SHORT {ticker1}, BUY {ticker2}"
        description = "Spread extremely above mean"
        action = "SHORT_SPREAD"
    else:
        signal = "NO TRADE"
        description = "Spread within normal range"
        action = "WAIT"

    # Quality metrics
    quality = "GOOD" if correlation > 0.8 else "MODERATE" if correlation > 0.6 else "POOR"

    return {
        "ticker1": ticker1,
        "ticker2": ticker2,
        "price1": round(prices1[-1], 2),
        "price2": round(prices2[-1], 2),
        "correlation": round(correlation, 3),
        "quality": quality,
        "current_spread": round(current_spread, 2),
        "spread_mean": round(spread_mean, 2),
        "spread_std": round(spread_std, 2),
        "spread_zscore": round(current_spread_z, 2),
        "signal": signal,
        "description": description,
        "action": action,
        "lookback": lookback
    }


def print_pair_analysis(analysis: Dict):
    """Print formatted pair analysis."""
    print("\n" + "="*60)
    print(f"PAIRS ANALYSIS: {analysis['ticker1']} / {analysis['ticker2']}")
    print("="*60)

    print(f"\n📊 CURRENT PRICES")
    print(f"   {analysis['ticker1']}:           ${analysis['price1']}")
    print(f"   {analysis['ticker2']}:           ${analysis['price2']}")

    print(f"\n📈 CORRELATION")
    print(f"   Correlation:      {analysis['correlation']}")
    print(f"   Quality:          {analysis['quality']}")

    print(f"\n📏 SPREAD ANALYSIS")
    print(f"   Current Spread:   ${analysis['current_spread']}")
    print(f"   Mean Spread:      ${analysis['spread_mean']}")
    print(f"   Spread Z-Score:   {analysis['spread_zscore']}")

    print(f"\n🎯 TRADING SIGNAL")
    print(f"   {analysis['signal']}")
    print(f"   {analysis['description']}")

    if analysis['action'] != "WAIT":
        print(f"\n💰 TRADE SETUP")
        if analysis['action'] == "LONG_SPREAD":
            print(f"   Long:  {analysis['ticker1']} (undervalued)")
            print(f"   Short: {analysis['ticker2']} (overvalued)")
        else:
            print(f"   Short: {analysis['ticker1']} (overvalued)")
            print(f"   Long:  {analysis['ticker2']} (undervalued)")
        print(f"   Target: Spread returns to ${analysis['spread_mean']}")

    if analysis['quality'] == "POOR":
        print(f"\n⚠️  WARNING: Low correlation ({analysis['correlation']})")
        print("   Pairs trading may not be suitable for this pair.")

    print("\n" + "="*60 + "\n")


def main():
    parser = argparse.ArgumentParser(description="Pairs Trading Analyzer")

    parser.add_argument("--pair", type=str, required=True,
                        help="Stock pair (e.g., KO,PEP)")
    parser.add_argument("--lookback", type=int, default=60,
                        help="Lookback period (default: 60)")

    args = parser.parse_args()

    tickers = [t.strip().upper() for t in args.pair.split(",")]
    if len(tickers) != 2:
        print("Error: Please provide exactly 2 tickers separated by comma")
        return

    analysis = analyze_pair(tickers[0], tickers[1], args.lookback)
    print_pair_analysis(analysis)


if __name__ == "__main__":
    main()
```

## References

### references/mean_reversion_guide.md

```markdown
# Mean Reversion Quick Reference

## Core Concept
Prices oscillate around a mean; extreme deviations revert.

## Bollinger Band Signals

| Price Position | Signal | Confirmation |
|----------------|--------|--------------|
| At lower band | BUY | RSI < 30 |
| At upper band | SELL | RSI > 70 |
| At middle band | EXIT | Take profits |

## Z-Score Signals

| Z-Score | Signal | Probability |
|---------|--------|-------------|
| < -3 | STRONG BUY | 99.7% return |
| < -2 | BUY | 95% return |
| -1 to +1 | NO TRADE | Normal range |
| > +2 | SELL | 95% return |
| > +3 | STRONG SELL | 99.7% return |

## When to Avoid Mean Reversion

1. **Band Squeeze**: Breakout pending
2. **Band Walk**: Price hugging band (trending)
3. **Expanding Bands**: Volatility increasing
4. **Fundamental Catalyst**: News driving price
5. **52-Week Extremes**: May not revert

## Pairs Trading Checklist

- [ ] Correlation > 0.8
- [ ] Same sector
- [ ] Spread Z-score > |2|
- [ ] No company-specific news
- [ ] Similar market cap/beta

## Risk Management

| Rule | Value |
|------|-------|
| Risk per trade | 1-2% |
| Stop loss | Below lower band / above upper |
| Time stop | 5-10 days |
| Target 1 | Middle band |
| Target 2 | Opposite band |
```

## File Extraction Rules

1. **Scripts**: Extract after `### scripts/{filename}` → `scripts/`
2. **References**: Extract after `### references/{filename}` → `references/`
3. Create directories if needed
4. Make scripts executable

## Post-Init Steps

1. Create directories:
   ```bash
   mkdir -p scripts references
   ```

2. Extract files from code blocks

3. Install dependencies:
   ```bash
   pip install pandas numpy yfinance scipy
   ```

4. Test Bollinger analyzer:
   ```bash
   python scripts/bollinger_analyzer.py --ticker AAPL
   ```

5. Test Z-score calculator:
   ```bash
   python scripts/zscore_calculator.py --ticker AAPL
   ```

6. Test pairs analyzer:
   ```bash
   python scripts/pairs_analyzer.py --pair KO,PEP
   ```

## Compatibility

Tested with: Claude, ChatGPT, Gemini, Copilot

---
Downloaded from [Find Skill.ai](https://findskill.ai)
