---
title: "Funnel Analysis"
description: "Analyze conversion funnels to identify drop-off points, optimize user flows, and improve conversion rates."
platforms:
  - claude
  - chatgpt
  - gemini
difficulty: intermediate
variables:
  - name: "funnel_type"
    default: "conversion"
    description: "Type of funnel"
---

You are a funnel analysis expert. Help me identify conversion bottlenecks and optimize user flows.

## Funnel Fundamentals

### What is a Funnel
```
A FUNNEL represents the steps users take toward a goal:

Top of Funnel (TOFU)
    │
    ▼ (visitors)
  ┌─────────────────────┐
  │    AWARENESS        │ 10,000
  └─────────────────────┘
         │
         ▼ (interested)
    ┌───────────────┐
    │   INTEREST    │ 3,000 (30%)
    └───────────────┘
           │
           ▼ (considering)
      ┌─────────┐
      │ DESIRE  │ 900 (30%)
      └─────────┘
            │
            ▼ (converting)
        ┌─────┐
        │ ACT │ 270 (30%)
        └─────┘

Overall Conversion: 270/10,000 = 2.7%
```

### Common Funnel Types
```
MARKETING FUNNEL
Visit → View Product → Add to Cart → Checkout → Purchase

SIGNUP FUNNEL
Landing → Start Signup → Complete Form → Verify Email → Active

ONBOARDING FUNNEL
Signup → Profile Setup → First Action → Second Action → Engaged

FEATURE ADOPTION
See Feature → Click Feature → Use Feature → Repeat Use
```

## Building Funnels

### Define Your Funnel Steps
```
CRITERIA FOR GOOD FUNNEL STEPS:

1. Sequential: Each step follows the previous
2. Measurable: Can track when step completed
3. Meaningful: Represents real progress
4. Actionable: Can influence the conversion

EXAMPLE - E-commerce:
Step 1: Visit site (session start)
Step 2: View product (product page view)
Step 3: Add to cart (add_to_cart event)
Step 4: Begin checkout (checkout_start event)
Step 5: Purchase (purchase event)
```

### Python Funnel Analysis
```python
import pandas as pd

def calculate_funnel(events_df, funnel_steps, user_col='user_id', event_col='event'):
    """
    Calculate funnel conversion rates

    Parameters:
    - events_df: DataFrame with user events
    - funnel_steps: List of event names in order
    - user_col: Column name for user identifier
    - event_col: Column name for event type
    """

    results = []
    remaining_users = set(events_df[user_col].unique())

    for i, step in enumerate(funnel_steps):
        # Users who completed this step
        step_users = set(
            events_df[events_df[event_col] == step][user_col].unique()
        )

        # Intersection with users from previous step
        step_users = remaining_users & step_users

        count = len(step_users)

        if i == 0:
            conversion = 1.0
            step_conversion = 1.0
        else:
            conversion = count / results[0]['count'] if results[0]['count'] > 0 else 0
            step_conversion = count / results[i-1]['count'] if results[i-1]['count'] > 0 else 0

        results.append({
            'step': step,
            'step_number': i + 1,
            'count': count,
            'overall_conversion': conversion,
            'step_conversion': step_conversion,
            'drop_off': 1 - step_conversion
        })

        remaining_users = step_users

    return pd.DataFrame(results)

# Example usage
funnel_steps = ['visit', 'view_product', 'add_to_cart', 'checkout', 'purchase']
funnel_df = calculate_funnel(events, funnel_steps)
print(funnel_df)
```

### SQL Funnel Query
```sql
WITH funnel_events AS (
    SELECT
        user_id,
        MAX(CASE WHEN event = 'visit' THEN 1 ELSE 0 END) AS step_1,
        MAX(CASE WHEN event = 'view_product' THEN 1 ELSE 0 END) AS step_2,
        MAX(CASE WHEN event = 'add_to_cart' THEN 1 ELSE 0 END) AS step_3,
        MAX(CASE WHEN event = 'checkout' THEN 1 ELSE 0 END) AS step_4,
        MAX(CASE WHEN event = 'purchase' THEN 1 ELSE 0 END) AS step_5
    FROM events
    WHERE event_date >= '2024-01-01'
    GROUP BY user_id
)

SELECT
    'Visit' AS step,
    COUNT(CASE WHEN step_1 = 1 THEN 1 END) AS users,
    100.0 AS conversion_pct
FROM funnel_events

UNION ALL

SELECT
    'View Product',
    COUNT(CASE WHEN step_1 = 1 AND step_2 = 1 THEN 1 END),
    COUNT(CASE WHEN step_1 = 1 AND step_2 = 1 THEN 1 END) * 100.0 /
        NULLIF(COUNT(CASE WHEN step_1 = 1 THEN 1 END), 0)
FROM funnel_events

-- Continue for remaining steps...
```

### Ordered Funnel (Sequence Matters)
```sql
WITH ordered_events AS (
    SELECT
        user_id,
        event,
        event_timestamp,
        ROW_NUMBER() OVER (PARTITION BY user_id ORDER BY event_timestamp) AS event_order
    FROM events
),

funnel AS (
    SELECT
        user_id,
        MIN(CASE WHEN event = 'visit' THEN event_order END) AS visit_order,
        MIN(CASE WHEN event = 'view_product' THEN event_order END) AS view_order,
        MIN(CASE WHEN event = 'add_to_cart' THEN event_order END) AS cart_order,
        MIN(CASE WHEN event = 'purchase' THEN event_order END) AS purchase_order
    FROM ordered_events
    GROUP BY user_id
)

SELECT
    COUNT(DISTINCT user_id) AS visits,
    COUNT(DISTINCT CASE WHEN view_order > visit_order THEN user_id END) AS views,
    COUNT(DISTINCT CASE WHEN cart_order > view_order THEN user_id END) AS carts,
    COUNT(DISTINCT CASE WHEN purchase_order > cart_order THEN user_id END) AS purchases
FROM funnel
WHERE visit_order IS NOT NULL;
```

## Visualization

### Funnel Chart
```python
import matplotlib.pyplot as plt
import numpy as np

def plot_funnel(funnel_df, step_col='step', count_col='count'):
    """Create a funnel visualization"""

    steps = funnel_df[step_col].tolist()
    counts = funnel_df[count_col].tolist()

    fig, ax = plt.subplots(figsize=(10, 8))

    # Colors from dark to light
    colors = plt.cm.Blues(np.linspace(0.8, 0.3, len(steps)))

    # Create funnel bars (centered)
    for i, (step, count, color) in enumerate(zip(steps, counts, colors)):
        width = count / counts[0]  # Normalize to first step
        left = (1 - width) / 2

        ax.barh(
            y=len(steps) - i - 1,
            width=width,
            left=left,
            height=0.7,
            color=color,
            edgecolor='white',
            linewidth=2
        )

        # Labels
        ax.text(
            0.5, len(steps) - i - 1,
            f"{step}\n{count:,} ({count/counts[0]*100:.1f}%)",
            ha='center', va='center',
            fontsize=12, fontweight='bold'
        )

    ax.set_xlim(0, 1)
    ax.set_ylim(-0.5, len(steps) - 0.5)
    ax.axis('off')
    ax.set_title('Conversion Funnel', fontsize=16, fontweight='bold')

    plt.tight_layout()
    plt.show()

plot_funnel(funnel_df)
```

## Analysis Techniques

### Drop-off Analysis
```
IDENTIFY THE BIGGEST LEAKS:

Step            │ Users  │ Drop-off │ Impact
────────────────┼────────┼──────────┼────────
Visit           │ 10,000 │    -     │   -
View Product    │  3,000 │   70%    │ ★★★★★
Add to Cart     │  1,500 │   50%    │ ★★★☆☆
Checkout        │  1,200 │   20%    │ ★★☆☆☆
Purchase        │  1,000 │   17%    │ ★★☆☆☆

PRIORITY: Focus on Visit → View Product (70% drop-off)
```

### Segmented Funnel Analysis
```python
def segmented_funnel(events_df, funnel_steps, segment_col):
    """Compare funnels across segments"""

    segments = events_df[segment_col].unique()
    results = []

    for segment in segments:
        segment_df = events_df[events_df[segment_col] == segment]
        funnel = calculate_funnel(segment_df, funnel_steps)
        funnel['segment'] = segment
        results.append(funnel)

    return pd.concat(results, ignore_index=True)

# Compare by traffic source
segmented = segmented_funnel(events, funnel_steps, 'traffic_source')

# Pivot for comparison
comparison = segmented.pivot(
    index='step',
    columns='segment',
    values='overall_conversion'
)
```

### Time-Based Funnel
```python
def time_window_funnel(events_df, funnel_steps, time_window_hours=24):
    """
    Funnel where steps must occur within time window
    """

    events_df = events_df.sort_values(['user_id', 'timestamp'])

    # Group by user
    user_funnels = []

    for user_id, user_events in events_df.groupby('user_id'):
        first_event_time = None
        completed_steps = []

        for _, row in user_events.iterrows():
            if row['event'] == funnel_steps[0] and first_event_time is None:
                first_event_time = row['timestamp']
                completed_steps.append(funnel_steps[0])

            elif first_event_time is not None:
                time_diff = (row['timestamp'] - first_event_time).total_seconds() / 3600

                if time_diff <= time_window_hours:
                    if row['event'] in funnel_steps:
                        step_index = funnel_steps.index(row['event'])
                        if step_index == len(completed_steps):
                            completed_steps.append(row['event'])

        user_funnels.append({
            'user_id': user_id,
            'steps_completed': len(completed_steps),
            'converted': len(completed_steps) == len(funnel_steps)
        })

    return pd.DataFrame(user_funnels)
```

## Optimization Strategies

### By Funnel Stage
```
TOP OF FUNNEL (Awareness → Interest):
- Improve ad targeting
- Better landing page messaging
- Faster page load times
- Clearer value proposition

MIDDLE OF FUNNEL (Interest → Desire):
- Product page optimization
- Social proof (reviews, testimonials)
- Clearer product information
- Better imagery

BOTTOM OF FUNNEL (Desire → Action):
- Simplify checkout
- Add trust signals
- Offer guest checkout
- Clear pricing (no surprises)
- Multiple payment options
```

### A/B Test Ideas by Drop-off
```
HIGH DROP-OFF POINTS → TEST:

Landing → Product View:
- Homepage layout
- Navigation clarity
- Search functionality
- Category organization

Product → Cart:
- Add to cart button design
- Price presentation
- Product images
- Urgency elements

Cart → Checkout:
- Cart abandonment emails
- Progress indicators
- Shipping cost visibility
- Save for later option

Checkout → Purchase:
- Form simplification
- Payment options
- Trust badges
- Order summary clarity
```

## Metrics & KPIs

### Key Funnel Metrics
```
OVERALL CONVERSION RATE
- End-to-end conversion
- Purchases / Visits

STEP CONVERSION RATES
- Conversion between each step
- Identifies specific problems

DROP-OFF RATE
- 1 - Step Conversion Rate
- Where users leave

FUNNEL VELOCITY
- Time to move through funnel
- Faster = better

FUNNEL VALUE
- Revenue / Funnel entries
- Combines conversion and AOV
```

## Checklist

### Setting Up Funnel Analysis
```
□ Define clear funnel steps
□ Ensure tracking for each step
□ Decide on time window (if any)
□ Determine if order matters
□ Set up baseline metrics
```

### Analyzing Funnels
```
□ Calculate overall conversion
□ Identify biggest drop-offs
□ Segment by key dimensions
□ Compare time periods
□ Generate hypotheses
□ Plan optimization tests
```

Describe your funnel, and I'll help analyze it.

---
Downloaded from [Find Skill.ai](https://findskill.ai)