---
title: "React TypeScript Supabase Full-Stack"
description: "Build production-ready full-stack applications with React TypeScript frontend, Node.js Express backend, and Supabase for database, auth, storage, and real-time features."
platforms:
  - claude
  - chatgpt
  - gemini
  - copilot
difficulty: intermediate
variables:
  - name: "project_name"
    default: "my-fullstack-app"
    description: "Name of your full-stack project"
  - name: "backend_framework"
    default: "express"
    description: "Node.js backend framework (express, fastify)"
---

# React TypeScript Supabase Full-Stack Development

Build production-ready full-stack applications with React TypeScript frontend, Node.js Express backend, and Supabase for database, authentication, storage, and real-time features.

## Quick Start

### Project Structure

```
{{project_name}}/
├── frontend/                    # React TypeScript app
│   ├── src/
│   │   ├── components/
│   │   ├── hooks/
│   │   ├── lib/
│   │   │   └── supabase.ts     # Supabase client (anon key)
│   │   ├── types/
│   │   │   └── database.types.ts
│   │   └── App.tsx
│   └── package.json
├── backend/                     # Node.js Express API
│   ├── src/
│   │   ├── routes/
│   │   ├── middleware/
│   │   ├── services/
│   │   │   └── supabase.ts     # Supabase admin client (service role)
│   │   └── index.ts
│   └── package.json
└── supabase/
    └── migrations/
```

### Environment Variables

```bash
# frontend/.env
VITE_SUPABASE_URL=https://your-project.supabase.co
VITE_SUPABASE_ANON_KEY=your-anon-key

# backend/.env
SUPABASE_URL=https://your-project.supabase.co
SUPABASE_SERVICE_ROLE_KEY=your-service-role-key  # NEVER expose this!
SUPABASE_JWT_SECRET=your-jwt-secret
PORT=3001
```

---

## Generate TypeScript Types

Always generate types from your database schema:

```bash
# Install Supabase CLI
npm install -g supabase

# Login and link project
supabase login
supabase link --project-ref your-project-ref

# Generate types from remote database
npx supabase gen types typescript --project-id "your-project-id" > frontend/src/types/database.types.ts

# Or from local development database
npx supabase gen types typescript --local > frontend/src/types/database.types.ts
```

### Generated Types Structure

```typescript
// database.types.ts
export type Json = string | number | boolean | null | { [key: string]: Json | undefined } | Json[]

export interface Database {
  public: {
    Tables: {
      users: {
        Row: {
          id: string
          email: string
          full_name: string | null
          avatar_url: string | null
          created_at: string
        }
        Insert: {
          id?: string
          email: string
          full_name?: string | null
          avatar_url?: string | null
          created_at?: string
        }
        Update: {
          id?: string
          email?: string
          full_name?: string | null
          avatar_url?: string | null
          created_at?: string
        }
      }
      posts: {
        Row: {
          id: string
          title: string
          content: string
          user_id: string
          published: boolean
          created_at: string
        }
        Insert: Omit<Database['public']['Tables']['posts']['Row'], 'id' | 'created_at'>
        Update: Partial<Database['public']['Tables']['posts']['Insert']>
      }
    }
    Views: {}
    Functions: {}
    Enums: {}
  }
}

// Helper types
export type Tables<T extends keyof Database['public']['Tables']> =
  Database['public']['Tables'][T]['Row']
export type InsertTables<T extends keyof Database['public']['Tables']> =
  Database['public']['Tables'][T]['Insert']
export type UpdateTables<T extends keyof Database['public']['Tables']> =
  Database['public']['Tables'][T]['Update']
```

---

## Frontend: React TypeScript with Supabase

### Supabase Client Setup

```typescript
// frontend/src/lib/supabase.ts
import { createClient } from '@supabase/supabase-js'
import type { Database } from '../types/database.types'

const supabaseUrl = import.meta.env.VITE_SUPABASE_URL
const supabaseAnonKey = import.meta.env.VITE_SUPABASE_ANON_KEY

if (!supabaseUrl || !supabaseAnonKey) {
  throw new Error('Missing Supabase environment variables')
}

export const supabase = createClient<Database>(supabaseUrl, supabaseAnonKey, {
  auth: {
    autoRefreshToken: true,
    persistSession: true,
    detectSessionInUrl: true
  }
})
```

### Authentication Context

```typescript
// frontend/src/contexts/AuthContext.tsx
import { createContext, useContext, useEffect, useState, ReactNode } from 'react'
import { User, Session, AuthError } from '@supabase/supabase-js'
import { supabase } from '../lib/supabase'

interface AuthContextType {
  user: User | null
  session: Session | null
  loading: boolean
  signUp: (email: string, password: string) => Promise<{ error: AuthError | null }>
  signIn: (email: string, password: string) => Promise<{ error: AuthError | null }>
  signInWithOAuth: (provider: 'google' | 'github') => Promise<void>
  signOut: () => Promise<void>
}

const AuthContext = createContext<AuthContextType | undefined>(undefined)

export function AuthProvider({ children }: { children: ReactNode }) {
  const [user, setUser] = useState<User | null>(null)
  const [session, setSession] = useState<Session | null>(null)
  const [loading, setLoading] = useState(true)

  useEffect(() => {
    // Get initial session
    supabase.auth.getSession().then(({ data: { session } }) => {
      setSession(session)
      setUser(session?.user ?? null)
      setLoading(false)
    })

    // Listen for auth changes
    const { data: { subscription } } = supabase.auth.onAuthStateChange(
      async (event, session) => {
        setSession(session)
        setUser(session?.user ?? null)
        setLoading(false)

        if (event === 'SIGNED_IN') {
          // Sync user profile with backend
          await fetch('/api/auth/sync', {
            method: 'POST',
            headers: {
              'Authorization': `Bearer ${session?.access_token}`,
              'Content-Type': 'application/json'
            }
          })
        }
      }
    )

    return () => subscription.unsubscribe()
  }, [])

  const signUp = async (email: string, password: string) => {
    const { error } = await supabase.auth.signUp({
      email,
      password,
      options: {
        emailRedirectTo: `${window.location.origin}/auth/callback`
      }
    })
    return { error }
  }

  const signIn = async (email: string, password: string) => {
    const { error } = await supabase.auth.signInWithPassword({ email, password })
    return { error }
  }

  const signInWithOAuth = async (provider: 'google' | 'github') => {
    await supabase.auth.signInWithOAuth({
      provider,
      options: {
        redirectTo: `${window.location.origin}/auth/callback`
      }
    })
  }

  const signOut = async () => {
    await supabase.auth.signOut()
  }

  return (
    <AuthContext.Provider value={{
      user,
      session,
      loading,
      signUp,
      signIn,
      signInWithOAuth,
      signOut
    }}>
      {children}
    </AuthContext.Provider>
  )
}

export function useAuth() {
  const context = useContext(AuthContext)
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider')
  }
  return context
}
```

### Custom Hooks for Data Fetching

```typescript
// frontend/src/hooks/usePosts.ts
import { useState, useEffect, useCallback } from 'react'
import { supabase } from '../lib/supabase'
import type { Tables, InsertTables, UpdateTables } from '../types/database.types'

type Post = Tables<'posts'>
type PostInsert = InsertTables<'posts'>
type PostUpdate = UpdateTables<'posts'>

interface UsePostsReturn {
  posts: Post[]
  loading: boolean
  error: Error | null
  createPost: (post: PostInsert) => Promise<Post | null>
  updatePost: (id: string, updates: PostUpdate) => Promise<Post | null>
  deletePost: (id: string) => Promise<boolean>
  refetch: () => Promise<void>
}

export function usePosts(userId?: string): UsePostsReturn {
  const [posts, setPosts] = useState<Post[]>([])
  const [loading, setLoading] = useState(true)
  const [error, setError] = useState<Error | null>(null)

  const fetchPosts = useCallback(async () => {
    try {
      setLoading(true)
      let query = supabase
        .from('posts')
        .select('*')
        .order('created_at', { ascending: false })

      if (userId) {
        query = query.eq('user_id', userId)
      } else {
        query = query.eq('published', true)
      }

      const { data, error } = await query

      if (error) throw error
      setPosts(data ?? [])
    } catch (err) {
      setError(err instanceof Error ? err : new Error('Failed to fetch posts'))
    } finally {
      setLoading(false)
    }
  }, [userId])

  useEffect(() => {
    fetchPosts()
  }, [fetchPosts])

  const createPost = async (post: PostInsert): Promise<Post | null> => {
    const { data, error } = await supabase
      .from('posts')
      .insert(post)
      .select()
      .single()

    if (error) {
      setError(error)
      return null
    }

    setPosts(prev => [data, ...prev])
    return data
  }

  const updatePost = async (id: string, updates: PostUpdate): Promise<Post | null> => {
    const { data, error } = await supabase
      .from('posts')
      .update(updates)
      .eq('id', id)
      .select()
      .single()

    if (error) {
      setError(error)
      return null
    }

    setPosts(prev => prev.map(p => p.id === id ? data : p))
    return data
  }

  const deletePost = async (id: string): Promise<boolean> => {
    const { error } = await supabase
      .from('posts')
      .delete()
      .eq('id', id)

    if (error) {
      setError(error)
      return false
    }

    setPosts(prev => prev.filter(p => p.id !== id))
    return true
  }

  return { posts, loading, error, createPost, updatePost, deletePost, refetch: fetchPosts }
}
```

### Real-Time Subscriptions Hook

```typescript
// frontend/src/hooks/useRealtimeSubscription.ts
import { useEffect, useRef } from 'react'
import { RealtimeChannel, RealtimePostgresChangesPayload } from '@supabase/supabase-js'
import { supabase } from '../lib/supabase'
import type { Database } from '../types/database.types'

type TableName = keyof Database['public']['Tables']
type RowType<T extends TableName> = Database['public']['Tables'][T]['Row']

interface UseRealtimeOptions<T extends TableName> {
  table: T
  schema?: string
  event?: 'INSERT' | 'UPDATE' | 'DELETE' | '*'
  filter?: string
  onInsert?: (payload: RowType<T>) => void
  onUpdate?: (payload: { old: RowType<T>; new: RowType<T> }) => void
  onDelete?: (payload: RowType<T>) => void
  onChange?: (payload: RealtimePostgresChangesPayload<RowType<T>>) => void
}

export function useRealtimeSubscription<T extends TableName>(
  options: UseRealtimeOptions<T>
) {
  const channelRef = useRef<RealtimeChannel | null>(null)

  useEffect(() => {
    const {
      table,
      schema = 'public',
      event = '*',
      filter,
      onInsert,
      onUpdate,
      onDelete,
      onChange
    } = options

    const channelName = `realtime:${schema}:${table}:${filter ?? 'all'}`

    channelRef.current = supabase
      .channel(channelName)
      .on(
        'postgres_changes',
        {
          event,
          schema,
          table,
          filter
        },
        (payload) => {
          onChange?.(payload as RealtimePostgresChangesPayload<RowType<T>>)

          switch (payload.eventType) {
            case 'INSERT':
              onInsert?.(payload.new as RowType<T>)
              break
            case 'UPDATE':
              onUpdate?.({
                old: payload.old as RowType<T>,
                new: payload.new as RowType<T>
              })
              break
            case 'DELETE':
              onDelete?.(payload.old as RowType<T>)
              break
          }
        }
      )
      .subscribe()

    return () => {
      if (channelRef.current) {
        supabase.removeChannel(channelRef.current)
      }
    }
  }, [options.table, options.filter, options.event])

  return channelRef.current
}

// Usage example:
// useRealtimeSubscription({
//   table: 'posts',
//   event: '*',
//   onInsert: (newPost) => setPosts(prev => [newPost, ...prev]),
//   onUpdate: ({ new: updated }) => setPosts(prev =>
//     prev.map(p => p.id === updated.id ? updated : p)
//   ),
//   onDelete: (deleted) => setPosts(prev =>
//     prev.filter(p => p.id !== deleted.id)
//   )
// })
```

### File Upload Hook

```typescript
// frontend/src/hooks/useFileUpload.ts
import { useState } from 'react'
import { supabase } from '../lib/supabase'

interface UploadOptions {
  bucket: string
  path: string
  file: File
  onProgress?: (progress: number) => void
}

interface UseFileUploadReturn {
  uploading: boolean
  progress: number
  error: Error | null
  upload: (options: UploadOptions) => Promise<string | null>
  getPublicUrl: (bucket: string, path: string) => string
}

export function useFileUpload(): UseFileUploadReturn {
  const [uploading, setUploading] = useState(false)
  const [progress, setProgress] = useState(0)
  const [error, setError] = useState<Error | null>(null)

  const upload = async ({ bucket, path, file, onProgress }: UploadOptions): Promise<string | null> => {
    try {
      setUploading(true)
      setProgress(0)
      setError(null)

      const fileExt = file.name.split('.').pop()
      const filePath = `${path}/${Date.now()}.${fileExt}`

      const { error: uploadError } = await supabase.storage
        .from(bucket)
        .upload(filePath, file, {
          cacheControl: '3600',
          upsert: false
        })

      if (uploadError) throw uploadError

      setProgress(100)
      onProgress?.(100)

      return filePath
    } catch (err) {
      setError(err instanceof Error ? err : new Error('Upload failed'))
      return null
    } finally {
      setUploading(false)
    }
  }

  const getPublicUrl = (bucket: string, path: string): string => {
    const { data } = supabase.storage.from(bucket).getPublicUrl(path)
    return data.publicUrl
  }

  return { uploading, progress, error, upload, getPublicUrl }
}
```

---

## Backend: Node.js Express with Supabase Admin

### Supabase Admin Client (Service Role)

```typescript
// backend/src/services/supabase.ts
import { createClient } from '@supabase/supabase-js'
import type { Database } from '../types/database.types'

const supabaseUrl = process.env.SUPABASE_URL!
const supabaseServiceRoleKey = process.env.SUPABASE_SERVICE_ROLE_KEY!

if (!supabaseUrl || !supabaseServiceRoleKey) {
  throw new Error('Missing Supabase environment variables')
}

// Admin client - bypasses RLS!
// Only use in trusted server-side code
export const supabaseAdmin = createClient<Database>(
  supabaseUrl,
  supabaseServiceRoleKey,
  {
    auth: {
      autoRefreshToken: false,
      persistSession: false
    }
  }
)

// Create a client that respects RLS using user's JWT
export function createUserClient(accessToken: string) {
  return createClient<Database>(supabaseUrl, process.env.SUPABASE_ANON_KEY!, {
    global: {
      headers: {
        Authorization: `Bearer ${accessToken}`
      }
    },
    auth: {
      autoRefreshToken: false,
      persistSession: false
    }
  })
}
```

### JWT Authentication Middleware

```typescript
// backend/src/middleware/auth.ts
import { Request, Response, NextFunction } from 'express'
import jwt from 'jsonwebtoken'
import { supabaseAdmin } from '../services/supabase'

interface JWTPayload {
  sub: string
  email: string
  role: string
  aud: string
  exp: number
}

declare global {
  namespace Express {
    interface Request {
      user?: {
        id: string
        email: string
        role: string
      }
      accessToken?: string
    }
  }
}

export async function authMiddleware(
  req: Request,
  res: Response,
  next: NextFunction
) {
  try {
    const authHeader = req.headers.authorization

    if (!authHeader?.startsWith('Bearer ')) {
      return res.status(401).json({ error: 'Missing authorization header' })
    }

    const token = authHeader.split(' ')[1]

    // Verify JWT with Supabase's JWT secret
    const decoded = jwt.verify(
      token,
      process.env.SUPABASE_JWT_SECRET!
    ) as JWTPayload

    // Check token expiration
    if (decoded.exp * 1000 < Date.now()) {
      return res.status(401).json({ error: 'Token expired' })
    }

    // Optionally verify user still exists in Supabase
    const { data: user, error } = await supabaseAdmin.auth.admin.getUserById(
      decoded.sub
    )

    if (error || !user) {
      return res.status(401).json({ error: 'User not found' })
    }

    req.user = {
      id: decoded.sub,
      email: decoded.email,
      role: decoded.role
    }
    req.accessToken = token

    next()
  } catch (error) {
    console.error('Auth error:', error)
    return res.status(401).json({ error: 'Invalid token' })
  }
}

// Optional: Role-based access control
export function requireRole(...roles: string[]) {
  return (req: Request, res: Response, next: NextFunction) => {
    if (!req.user) {
      return res.status(401).json({ error: 'Not authenticated' })
    }

    if (!roles.includes(req.user.role)) {
      return res.status(403).json({ error: 'Insufficient permissions' })
    }

    next()
  }
}
```

### Express API Routes

```typescript
// backend/src/routes/posts.ts
import { Router, Request, Response } from 'express'
import { supabaseAdmin, createUserClient } from '../services/supabase'
import { authMiddleware } from '../middleware/auth'

const router = Router()

// Public: Get published posts
router.get('/', async (req: Request, res: Response) => {
  try {
    const { data, error } = await supabaseAdmin
      .from('posts')
      .select(`
        *,
        author:users(id, full_name, avatar_url)
      `)
      .eq('published', true)
      .order('created_at', { ascending: false })
      .limit(20)

    if (error) throw error
    res.json(data)
  } catch (error) {
    console.error('Error fetching posts:', error)
    res.status(500).json({ error: 'Failed to fetch posts' })
  }
})

// Protected: Create post
router.post('/', authMiddleware, async (req: Request, res: Response) => {
  try {
    const { title, content, published = false } = req.body

    if (!title || !content) {
      return res.status(400).json({ error: 'Title and content are required' })
    }

    // Use user client to respect RLS
    const userClient = createUserClient(req.accessToken!)

    const { data, error } = await userClient
      .from('posts')
      .insert({
        title,
        content,
        published,
        user_id: req.user!.id
      })
      .select()
      .single()

    if (error) throw error
    res.status(201).json(data)
  } catch (error) {
    console.error('Error creating post:', error)
    res.status(500).json({ error: 'Failed to create post' })
  }
})

// Protected: Update post
router.patch('/:id', authMiddleware, async (req: Request, res: Response) => {
  try {
    const { id } = req.params
    const { title, content, published } = req.body

    const userClient = createUserClient(req.accessToken!)

    const { data, error } = await userClient
      .from('posts')
      .update({ title, content, published })
      .eq('id', id)
      .select()
      .single()

    if (error) throw error
    if (!data) {
      return res.status(404).json({ error: 'Post not found' })
    }

    res.json(data)
  } catch (error) {
    console.error('Error updating post:', error)
    res.status(500).json({ error: 'Failed to update post' })
  }
})

// Protected: Delete post
router.delete('/:id', authMiddleware, async (req: Request, res: Response) => {
  try {
    const { id } = req.params
    const userClient = createUserClient(req.accessToken!)

    const { error } = await userClient
      .from('posts')
      .delete()
      .eq('id', id)

    if (error) throw error
    res.status(204).send()
  } catch (error) {
    console.error('Error deleting post:', error)
    res.status(500).json({ error: 'Failed to delete post' })
  }
})

// Admin: Bulk operations (bypasses RLS)
router.post('/admin/bulk-publish', authMiddleware, async (req: Request, res: Response) => {
  if (req.user?.role !== 'admin') {
    return res.status(403).json({ error: 'Admin access required' })
  }

  try {
    const { postIds } = req.body

    // Using admin client to bypass RLS
    const { data, error } = await supabaseAdmin
      .from('posts')
      .update({ published: true })
      .in('id', postIds)
      .select()

    if (error) throw error
    res.json({ updated: data?.length ?? 0 })
  } catch (error) {
    console.error('Error bulk publishing:', error)
    res.status(500).json({ error: 'Failed to bulk publish' })
  }
})

export default router
```

### Express App Setup

```typescript
// backend/src/index.ts
import express from 'express'
import cors from 'cors'
import helmet from 'helmet'
import rateLimit from 'express-rate-limit'
import postsRouter from './routes/posts'
import usersRouter from './routes/users'
import authRouter from './routes/auth'

const app = express()

// Security middleware
app.use(helmet())
app.use(cors({
  origin: process.env.FRONTEND_URL || 'http://localhost:5173',
  credentials: true
}))

// Rate limiting
const limiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100, // limit each IP to 100 requests per windowMs
  message: { error: 'Too many requests, please try again later' }
})
app.use('/api/', limiter)

// Body parsing
app.use(express.json({ limit: '10mb' }))

// Routes
app.use('/api/auth', authRouter)
app.use('/api/posts', postsRouter)
app.use('/api/users', usersRouter)

// Health check
app.get('/health', (req, res) => {
  res.json({ status: 'ok', timestamp: new Date().toISOString() })
})

// Error handler
app.use((err: Error, req: express.Request, res: express.Response, next: express.NextFunction) => {
  console.error('Unhandled error:', err)
  res.status(500).json({ error: 'Internal server error' })
})

const PORT = process.env.PORT || 3001
app.listen(PORT, () => {
  console.log(`Server running on port ${PORT}`)
})
```

---

## Database Schema and RLS Policies

### SQL Migrations

```sql
-- migrations/001_initial_schema.sql

-- Users table (extends Supabase auth.users)
CREATE TABLE public.users (
  id UUID REFERENCES auth.users(id) ON DELETE CASCADE PRIMARY KEY,
  email TEXT NOT NULL,
  full_name TEXT,
  avatar_url TEXT,
  role TEXT DEFAULT 'user' CHECK (role IN ('user', 'admin')),
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- Posts table
CREATE TABLE public.posts (
  id UUID DEFAULT gen_random_uuid() PRIMARY KEY,
  title TEXT NOT NULL,
  content TEXT NOT NULL,
  user_id UUID REFERENCES public.users(id) ON DELETE CASCADE NOT NULL,
  published BOOLEAN DEFAULT FALSE,
  created_at TIMESTAMPTZ DEFAULT NOW(),
  updated_at TIMESTAMPTZ DEFAULT NOW()
);

-- Indexes
CREATE INDEX idx_posts_user_id ON public.posts(user_id);
CREATE INDEX idx_posts_published ON public.posts(published) WHERE published = true;
CREATE INDEX idx_posts_created_at ON public.posts(created_at DESC);

-- Updated at trigger
CREATE OR REPLACE FUNCTION update_updated_at()
RETURNS TRIGGER AS $$
BEGIN
  NEW.updated_at = NOW();
  RETURN NEW;
END;
$$ LANGUAGE plpgsql;

CREATE TRIGGER update_users_updated_at
  BEFORE UPDATE ON public.users
  FOR EACH ROW EXECUTE FUNCTION update_updated_at();

CREATE TRIGGER update_posts_updated_at
  BEFORE UPDATE ON public.posts
  FOR EACH ROW EXECUTE FUNCTION update_updated_at();

-- Auto-create user profile on signup
CREATE OR REPLACE FUNCTION handle_new_user()
RETURNS TRIGGER AS $$
BEGIN
  INSERT INTO public.users (id, email, full_name, avatar_url)
  VALUES (
    NEW.id,
    NEW.email,
    NEW.raw_user_meta_data->>'full_name',
    NEW.raw_user_meta_data->>'avatar_url'
  );
  RETURN NEW;
END;
$$ LANGUAGE plpgsql SECURITY DEFINER;

CREATE TRIGGER on_auth_user_created
  AFTER INSERT ON auth.users
  FOR EACH ROW EXECUTE FUNCTION handle_new_user();
```

### Row Level Security Policies

```sql
-- migrations/002_rls_policies.sql

-- Enable RLS on all tables
ALTER TABLE public.users ENABLE ROW LEVEL SECURITY;
ALTER TABLE public.posts ENABLE ROW LEVEL SECURITY;

-- Users policies
CREATE POLICY "Users can view all profiles"
  ON public.users FOR SELECT
  TO authenticated
  USING (true);

CREATE POLICY "Users can update own profile"
  ON public.users FOR UPDATE
  TO authenticated
  USING (auth.uid() = id)
  WITH CHECK (auth.uid() = id);

-- Posts policies
CREATE POLICY "Anyone can view published posts"
  ON public.posts FOR SELECT
  USING (published = true);

CREATE POLICY "Authenticated users can view own posts"
  ON public.posts FOR SELECT
  TO authenticated
  USING (auth.uid() = user_id);

CREATE POLICY "Users can create own posts"
  ON public.posts FOR INSERT
  TO authenticated
  WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can update own posts"
  ON public.posts FOR UPDATE
  TO authenticated
  USING (auth.uid() = user_id)
  WITH CHECK (auth.uid() = user_id);

CREATE POLICY "Users can delete own posts"
  ON public.posts FOR DELETE
  TO authenticated
  USING (auth.uid() = user_id);

-- Admin policies (using role from JWT)
CREATE POLICY "Admins can do anything with posts"
  ON public.posts
  TO authenticated
  USING (
    (SELECT role FROM public.users WHERE id = auth.uid()) = 'admin'
  );
```

### Storage Policies

```sql
-- Storage bucket for avatars
INSERT INTO storage.buckets (id, name, public)
VALUES ('avatars', 'avatars', true);

-- Storage policies
CREATE POLICY "Avatar images are publicly accessible"
  ON storage.objects FOR SELECT
  USING (bucket_id = 'avatars');

CREATE POLICY "Users can upload their own avatar"
  ON storage.objects FOR INSERT
  TO authenticated
  WITH CHECK (
    bucket_id = 'avatars' AND
    (storage.foldername(name))[1] = auth.uid()::text
  );

CREATE POLICY "Users can update their own avatar"
  ON storage.objects FOR UPDATE
  TO authenticated
  USING (
    bucket_id = 'avatars' AND
    (storage.foldername(name))[1] = auth.uid()::text
  );

CREATE POLICY "Users can delete their own avatar"
  ON storage.objects FOR DELETE
  TO authenticated
  USING (
    bucket_id = 'avatars' AND
    (storage.foldername(name))[1] = auth.uid()::text
  );
```

---

## Error Handling Patterns

### Frontend Error Boundary

```typescript
// frontend/src/components/ErrorBoundary.tsx
import { Component, ErrorInfo, ReactNode } from 'react'

interface Props {
  children: ReactNode
  fallback?: ReactNode
}

interface State {
  hasError: boolean
  error?: Error
}

export class ErrorBoundary extends Component<Props, State> {
  constructor(props: Props) {
    super(props)
    this.state = { hasError: false }
  }

  static getDerivedStateFromError(error: Error): State {
    return { hasError: true, error }
  }

  componentDidCatch(error: Error, errorInfo: ErrorInfo) {
    console.error('Error caught by boundary:', error, errorInfo)
    // Log to error tracking service
  }

  render() {
    if (this.state.hasError) {
      return this.props.fallback || (
        <div className="error-container">
          <h2>Something went wrong</h2>
          <button onClick={() => this.setState({ hasError: false })}>
            Try again
          </button>
        </div>
      )
    }

    return this.props.children
  }
}
```

### Supabase Error Types

```typescript
// frontend/src/utils/errors.ts
import { PostgrestError, AuthError } from '@supabase/supabase-js'

export function handleSupabaseError(error: PostgrestError | AuthError | null): string {
  if (!error) return ''

  // Auth errors
  if ('status' in error && error.status === 400) {
    switch (error.message) {
      case 'Invalid login credentials':
        return 'Email or password is incorrect'
      case 'Email not confirmed':
        return 'Please verify your email address'
      case 'User already registered':
        return 'An account with this email already exists'
      default:
        return error.message
    }
  }

  // Database errors
  if ('code' in error) {
    switch (error.code) {
      case '23505': // unique_violation
        return 'This record already exists'
      case '23503': // foreign_key_violation
        return 'Related record not found'
      case '42501': // insufficient_privilege
        return 'You do not have permission to perform this action'
      case 'PGRST301': // Row-level security violation
        return 'Access denied'
      default:
        return error.message
    }
  }

  return 'An unexpected error occurred'
}
```

---

## Testing Patterns

### Backend Integration Tests

```typescript
// backend/src/__tests__/posts.test.ts
import request from 'supertest'
import app from '../index'
import { supabaseAdmin } from '../services/supabase'

describe('Posts API', () => {
  let testUserId: string
  let authToken: string

  beforeAll(async () => {
    // Create test user
    const { data } = await supabaseAdmin.auth.admin.createUser({
      email: 'test@example.com',
      password: 'testpassword123',
      email_confirm: true
    })
    testUserId = data.user!.id

    // Get auth token
    const { data: session } = await supabaseAdmin.auth.signInWithPassword({
      email: 'test@example.com',
      password: 'testpassword123'
    })
    authToken = session.session!.access_token
  })

  afterAll(async () => {
    // Cleanup
    await supabaseAdmin.auth.admin.deleteUser(testUserId)
  })

  describe('GET /api/posts', () => {
    it('returns published posts', async () => {
      const res = await request(app)
        .get('/api/posts')
        .expect(200)

      expect(Array.isArray(res.body)).toBe(true)
    })
  })

  describe('POST /api/posts', () => {
    it('requires authentication', async () => {
      await request(app)
        .post('/api/posts')
        .send({ title: 'Test', content: 'Content' })
        .expect(401)
    })

    it('creates post when authenticated', async () => {
      const res = await request(app)
        .post('/api/posts')
        .set('Authorization', `Bearer ${authToken}`)
        .send({ title: 'Test Post', content: 'Test content' })
        .expect(201)

      expect(res.body.title).toBe('Test Post')
      expect(res.body.user_id).toBe(testUserId)
    })
  })
})
```

---

## Deployment Checklist

### Environment Variables
- [ ] `SUPABASE_URL` - Your Supabase project URL
- [ ] `SUPABASE_ANON_KEY` - Public anon key (frontend)
- [ ] `SUPABASE_SERVICE_ROLE_KEY` - Service role key (backend only!)
- [ ] `SUPABASE_JWT_SECRET` - JWT secret for token verification
- [ ] `DATABASE_URL` - Direct database connection (optional)

### Security Checklist
- [ ] RLS enabled on all tables
- [ ] Service role key only in backend
- [ ] CORS configured correctly
- [ ] Rate limiting enabled
- [ ] Input validation on all endpoints
- [ ] Error messages don't leak sensitive info

### Performance Checklist
- [ ] Database indexes on frequently queried columns
- [ ] Connection pooling configured (PgBouncer)
- [ ] Real-time subscriptions cleaned up on unmount
- [ ] Images optimized before upload to storage

---

When you describe your full-stack application needs, I'll help you implement the complete solution with React TypeScript, Node.js, and Supabase.

---
Downloaded from [Find Skill.ai](https://findskill.ai)
