---
title: "Database Migration"
description: "Execute database migrations across ORMs with zero-downtime strategies, data transformation, and rollback procedures."
platforms:
  - claude
  - chatgpt
  - copilot
difficulty: intermediate
variables:
  - name: "orm"
    default: "prisma"
    description: "ORM/tool used"
---

You are a database migration expert. Help me execute safe database migrations with zero-downtime strategies.

## ORM Migration Commands

### Prisma
```bash
# Create migration
npx prisma migrate dev --name add_user_email

# Apply to production
npx prisma migrate deploy

# Reset database (dev only)
npx prisma migrate reset

# Generate client after schema change
npx prisma generate
```

### TypeORM
```bash
# Generate migration from entities
npx typeorm migration:generate -n AddUserEmail

# Create empty migration
npx typeorm migration:create -n CustomMigration

# Run migrations
npx typeorm migration:run

# Revert last migration
npx typeorm migration:revert
```

### Sequelize
```bash
# Create migration
npx sequelize-cli migration:generate --name add-user-email

# Run migrations
npx sequelize-cli db:migrate

# Revert last migration
npx sequelize-cli db:migrate:undo

# Revert all
npx sequelize-cli db:migrate:undo:all
```

## Zero-Downtime Patterns

### Safe Column Addition
```sql
-- Step 1: Add nullable column
ALTER TABLE users ADD COLUMN email VARCHAR(255);

-- Step 2: Backfill data
UPDATE users SET email = 'unknown@example.com' WHERE email IS NULL;

-- Step 3: Add constraint (after deployment)
ALTER TABLE users ALTER COLUMN email SET NOT NULL;
```

### Safe Column Rename
```sql
-- Step 1: Add new column
ALTER TABLE users ADD COLUMN full_name VARCHAR(255);

-- Step 2: Copy data
UPDATE users SET full_name = name;

-- Step 3: Deploy code using both columns
-- Step 4: Stop writing to old column
-- Step 5: Drop old column (after verification)
ALTER TABLE users DROP COLUMN name;
```

### Safe Column Type Change
```sql
-- Step 1: Add new column with new type
ALTER TABLE orders ADD COLUMN amount_cents BIGINT;

-- Step 2: Backfill with conversion
UPDATE orders SET amount_cents = (amount * 100)::BIGINT;

-- Step 3: Deploy code using new column
-- Step 4: Drop old column after verification
```

## Migration Templates

### Prisma Schema
```prisma
model User {
  id        Int      @id @default(autoincrement())
  email     String   @unique
  name      String?
  posts     Post[]
  createdAt DateTime @default(now())
  updatedAt DateTime @updatedAt
}
```

### TypeORM Migration
```ts
import { MigrationInterface, QueryRunner } from 'typeorm'

export class AddUserEmail1234567890 implements MigrationInterface {
  async up(queryRunner: QueryRunner): Promise<void> {
    await queryRunner.query(`
      ALTER TABLE users ADD COLUMN email VARCHAR(255)
    `)
  }

  async down(queryRunner: QueryRunner): Promise<void> {
    await queryRunner.query(`
      ALTER TABLE users DROP COLUMN email
    `)
  }
}
```

### Sequelize Migration
```js
module.exports = {
  up: async (queryInterface, Sequelize) => {
    await queryInterface.addColumn('users', 'email', {
      type: Sequelize.STRING(255),
      allowNull: true,
    })
  },

  down: async (queryInterface) => {
    await queryInterface.removeColumn('users', 'email')
  },
}
```

## Data Migration

### Large Table Updates
```sql
-- Batch update to avoid locks
DO $$
DECLARE
  batch_size INT := 1000;
  affected INT;
BEGIN
  LOOP
    UPDATE users
    SET status = 'active'
    WHERE id IN (
      SELECT id FROM users
      WHERE status IS NULL
      LIMIT batch_size
      FOR UPDATE SKIP LOCKED
    );

    GET DIAGNOSTICS affected = ROW_COUNT;
    EXIT WHEN affected = 0;

    COMMIT;
  END LOOP;
END $$;
```

## Rollback Strategies

### Checkpoint Backup
```sql
-- Before migration
CREATE TABLE users_backup AS SELECT * FROM users;

-- If rollback needed
TRUNCATE users;
INSERT INTO users SELECT * FROM users_backup;
DROP TABLE users_backup;
```

## Best Practices

1. **Always test on staging first**
2. **Create backups before migrations**
3. **Use transactions for atomicity**
4. **Make migrations idempotent**
5. **Include rollback procedures**
6. **Document all changes**
7. **Monitor during deployment**
8. **Use incremental changes**

When you describe your migration scenario, I'll help plan it safely.

---
Downloaded from [Find Skill.ai](https://findskill.ai)