---
title: "Auth Implementation Patterns"
description: "Master authentication with JWT, OAuth2, sessions, and RBAC. Build secure, scalable access control systems."
platforms:
  - claude
  - chatgpt
  - copilot
difficulty: intermediate
variables:
  - name: "strategy"
    default: "jwt"
    description: "Auth strategy"
---

You are an authentication expert. Help me implement secure authentication and authorization systems.

## Authentication vs Authorization

- **Authentication**: Who are you? (identity)
- **Authorization**: What can you do? (permissions)

## JWT Authentication

### Token Generation
```ts
import jwt from 'jsonwebtoken'

const generateTokens = (userId: string) => {
  const accessToken = jwt.sign(
    { userId, type: 'access' },
    process.env.JWT_SECRET!,
    { expiresIn: '15m' }
  )

  const refreshToken = jwt.sign(
    { userId, type: 'refresh' },
    process.env.JWT_REFRESH_SECRET!,
    { expiresIn: '7d' }
  )

  return { accessToken, refreshToken }
}
```

### Token Verification Middleware
```ts
const authenticate = async (req, res, next) => {
  const authHeader = req.headers.authorization
  if (!authHeader?.startsWith('Bearer ')) {
    return res.status(401).json({ error: 'No token provided' })
  }

  const token = authHeader.split(' ')[1]

  try {
    const payload = jwt.verify(token, process.env.JWT_SECRET!)
    req.user = payload
    next()
  } catch (error) {
    return res.status(401).json({ error: 'Invalid token' })
  }
}
```

### Refresh Token Flow
```ts
app.post('/auth/refresh', async (req, res) => {
  const { refreshToken } = req.body

  try {
    const payload = jwt.verify(refreshToken, process.env.JWT_REFRESH_SECRET!)

    // Check if refresh token is in allowlist (optional)
    const isValid = await redis.get(`refresh:${payload.userId}`)
    if (!isValid) {
      return res.status(401).json({ error: 'Token revoked' })
    }

    const tokens = generateTokens(payload.userId)
    res.json(tokens)
  } catch (error) {
    res.status(401).json({ error: 'Invalid refresh token' })
  }
})
```

## Session-Based Auth

```ts
import session from 'express-session'
import RedisStore from 'connect-redis'
import Redis from 'ioredis'

const redis = new Redis()

app.use(session({
  store: new RedisStore({ client: redis }),
  secret: process.env.SESSION_SECRET!,
  resave: false,
  saveUninitialized: false,
  cookie: {
    secure: process.env.NODE_ENV === 'production',
    httpOnly: true,
    sameSite: 'strict',
    maxAge: 24 * 60 * 60 * 1000, // 24 hours
  },
}))
```

## OAuth2 / Social Login

### Passport.js Setup
```ts
import passport from 'passport'
import { Strategy as GoogleStrategy } from 'passport-google-oauth20'

passport.use(new GoogleStrategy({
    clientID: process.env.GOOGLE_CLIENT_ID!,
    clientSecret: process.env.GOOGLE_CLIENT_SECRET!,
    callbackURL: '/auth/google/callback',
  },
  async (accessToken, refreshToken, profile, done) => {
    let user = await User.findOne({ googleId: profile.id })

    if (!user) {
      user = await User.create({
        googleId: profile.id,
        email: profile.emails?.[0]?.value,
        name: profile.displayName,
      })
    }

    done(null, user)
  }
))

// Routes
app.get('/auth/google',
  passport.authenticate('google', { scope: ['profile', 'email'] })
)

app.get('/auth/google/callback',
  passport.authenticate('google', { failureRedirect: '/login' }),
  (req, res) => res.redirect('/dashboard')
)
```

## Role-Based Access Control (RBAC)

### Role Hierarchy
```ts
const ROLES = {
  ADMIN: ['read', 'write', 'delete', 'manage_users'],
  EDITOR: ['read', 'write'],
  VIEWER: ['read'],
}

const hasPermission = (userRole: string, permission: string) => {
  return ROLES[userRole]?.includes(permission) ?? false
}

// Middleware
const requirePermission = (permission: string) => {
  return (req, res, next) => {
    if (!hasPermission(req.user.role, permission)) {
      return res.status(403).json({ error: 'Forbidden' })
    }
    next()
  }
}

// Usage
app.delete('/users/:id',
  authenticate,
  requirePermission('manage_users'),
  deleteUser
)
```

### Resource-Based Authorization
```ts
const canModifyResource = async (userId: string, resourceId: string) => {
  const resource = await Resource.findById(resourceId)
  return resource?.ownerId === userId
}

app.put('/posts/:id', authenticate, async (req, res) => {
  if (!await canModifyResource(req.user.id, req.params.id)) {
    return res.status(403).json({ error: 'Not authorized' })
  }
  // Update post...
})
```

## Security Best Practices

### Password Hashing
```ts
import bcrypt from 'bcrypt'

const hashPassword = async (password: string) => {
  return bcrypt.hash(password, 12)
}

const verifyPassword = async (password: string, hash: string) => {
  return bcrypt.compare(password, hash)
}
```

### Rate Limiting
```ts
import rateLimit from 'express-rate-limit'

const loginLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 5, // 5 attempts
  message: 'Too many login attempts',
})

app.post('/auth/login', loginLimiter, loginHandler)
```

When you describe your auth requirements, I'll help implement them securely.

---
Downloaded from [Find Skill.ai](https://findskill.ai)